
import React, { useState, useRef, useEffect } from 'react';
import { Send, Loader2, MapPin, Globe, BrainCircuit, Paperclip, Zap, Bot, User } from 'lucide-react';
import { generateChatResponse } from '../services/geminiService';
import { ChatMessage, ModelType } from '../types';
import { authService } from '../services/authService';

const ChatInterface: React.FC = () => {
  const [input, setInput] = useState('');
  const [messages, setMessages] = useState<ChatMessage[]>([
    { id: '0', role: 'model', text: 'Hello! I am your Gemini-powered assistant. I can help you research topics using Google Search, find places with Maps, or perform complex reasoning with my Thinking mode. How can I help you today?', timestamp: Date.now() }
  ]);
  const [isLoading, setIsLoading] = useState(false);
  const [useSearch, setUseSearch] = useState(false);
  const [useMaps, setUseMaps] = useState(false);
  const [useThinking, setUseThinking] = useState(false);
  const scrollRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (scrollRef.current) {
      scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
    }
  }, [messages]);

  const handleSend = async () => {
    if (!input.trim()) return;

    const userMsg: ChatMessage = {
      id: Date.now().toString(),
      role: 'user',
      text: input,
      timestamp: Date.now()
    };

    setMessages(prev => [...prev, userMsg]);
    setInput('');
    setIsLoading(true);

    // Determine model based on settings
    let model = ModelType.FLASH; // Default 2.5 Flash
    if (useThinking) model = ModelType.PRO; // 3.0 Pro for thinking
    // Use Flash Lite if user requested speed (implied logic or specific toggle not in basic UI but supported)

    // Get Location if Maps is enabled
    let location = undefined;
    if (useMaps && navigator.geolocation) {
      try {
        const pos: any = await new Promise((resolve, reject) => {
          navigator.geolocation.getCurrentPosition(resolve, reject);
        });
        location = {
          latitude: pos.coords.latitude,
          longitude: pos.coords.longitude
        };
      } catch (e) {
        console.warn("Location permission denied, proceeding without precise location.");
      }
    }

    try {
      // Fetch API Keys logic
      const currentUser = authService.getCurrentUser();
      let apiKeys: string[] = [];
      if (currentUser) {
        const keys = await authService.getUserApiKeys(currentUser.uid);
        const enabledKeys = keys.filter((k: any) => (k.status === 'valid' || k.status === 'unchecked') && k.isEnabled !== false);
        apiKeys = enabledKeys.map(k => k.fullKey.trim());

        if (localStorage.getItem('sf_random_key_rotation') === 'true') {
          apiKeys = apiKeys.sort(() => Math.random() - 0.5);
        }
      }

      const history = messages.map(m => ({ role: m.role, parts: [{ text: m.text }] }));
      const response = await generateChatResponse(
        userMsg.text,
        model,
        history,
        [], // No attachments in simple chat for now
        useThinking,
        useSearch,
        useMaps,
        location,
        apiKeys // Pass for robust rotation
      );

      const botMsg: ChatMessage = {
        id: (Date.now() + 1).toString(),
        role: 'model',
        text: response.text,
        timestamp: Date.now(),
        groundingMetadata: response.groundingChunks,
        isThinking: useThinking
      };

      // Track credit usage
      if (currentUser) {
        authService.incrementUserStat('apiCallsThisMonth', 1);
        authService.incrementUserStat('modelUsage', 1, model);
      }

      setMessages(prev => [...prev, botMsg]);
    } catch (error) {
      const errorMsg: ChatMessage = {
        id: (Date.now() + 1).toString(),
        role: 'model',
        text: "Sorry, I encountered an error processing your request. Please try again.",
        timestamp: Date.now()
      };
      setMessages(prev => [...prev, errorMsg]);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="flex flex-col h-[calc(100vh-2rem)] max-w-5xl mx-auto bg-slate-900/50 border border-slate-800 rounded-2xl overflow-hidden backdrop-blur-sm">
      {/* Header / Toolbar */}
      <div className="p-4 border-b border-slate-800 flex items-center justify-between bg-slate-950/80">
        <div className="flex items-center gap-2">
          <Bot className="text-amber-400 w-5 h-5" />
          <h2 className="font-bold text-slate-200">Intelligence Hub</h2>
        </div>
        <div className="flex gap-2">
          <ToggleBtn
            active={useThinking}
            onClick={() => setUseThinking(!useThinking)}
            icon={BrainCircuit}
            label="Deep Think (Pro)"
            color="text-purple-400"
          />
          <ToggleBtn
            active={useSearch}
            onClick={() => setUseSearch(!useSearch)}
            icon={Globe}
            label="Search"
            color="text-blue-400"
          />
          <ToggleBtn
            active={useMaps}
            onClick={() => setUseMaps(!useMaps)}
            icon={MapPin}
            label="Maps"
            color="text-red-400"
          />
        </div>
      </div>

      {/* Messages Area */}
      <div className="flex-1 overflow-y-auto p-6 space-y-6" ref={scrollRef}>
        {messages.map((msg) => (
          <div key={msg.id} className={`flex ${msg.role === 'user' ? 'justify-end' : 'justify-start'}`}>
            <div className={`max-w-[80%] rounded-2xl p-5 ${msg.role === 'user'
              ? 'bg-amber-600 text-white rounded-tr-none'
              : 'bg-slate-800 text-slate-200 rounded-tl-none border border-slate-700'
              }`}>
              <div className="flex items-center gap-2 mb-2 opacity-50 text-xs uppercase tracking-wider font-bold">
                {msg.role === 'user' ? <User size={12} /> : <Bot size={12} />}
                {msg.role}
                {msg.isThinking && <span className="ml-2 flex items-center gap-1 text-purple-300"><BrainCircuit size={10} /> Thought Process</span>}
              </div>
              <div className="whitespace-pre-wrap leading-relaxed">
                {msg.text}
              </div>

              {/* Grounding Sources */}
              {msg.groundingMetadata && msg.groundingMetadata.length > 0 && (
                <div className="mt-4 pt-4 border-t border-white/10">
                  <p className="text-xs font-bold mb-2 opacity-70">Sources:</p>
                  <div className="flex flex-wrap gap-2">
                    {msg.groundingMetadata.map((chunk, idx) => {
                      if (chunk.web) {
                        return (
                          <a key={idx} href={chunk.web.uri} target="_blank" rel="noreferrer" className="flex items-center gap-1 px-2 py-1 bg-black/20 rounded hover:bg-black/40 transition text-xs truncate max-w-[200px]">
                            <Globe size={10} /> {chunk.web.title}
                          </a>
                        )
                      }
                      if (chunk.maps) {
                        return (
                          <a key={idx} href={chunk.maps.uri} target="_blank" rel="noreferrer" className="flex items-center gap-1 px-2 py-1 bg-black/20 rounded hover:bg-black/40 transition text-xs truncate max-w-[200px]">
                            <MapPin size={10} /> {chunk.maps.title}
                          </a>
                        )
                      }
                      return null;
                    })}
                  </div>
                </div>
              )}
            </div>
          </div>
        ))}
        {isLoading && (
          <div className="flex justify-start">
            <div className="bg-slate-800 rounded-2xl p-4 flex items-center gap-3 rounded-tl-none border border-slate-700">
              <Loader2 className="w-5 h-5 text-amber-400 animate-spin" />
              <span className="text-slate-400 text-sm">Gemini is processing...</span>
            </div>
          </div>
        )}
      </div>

      {/* Input Area */}
      <div className="p-4 bg-slate-950 border-t border-slate-800">
        <div className="relative flex items-center">
          <input
            type="text"
            value={input}
            onChange={(e) => setInput(e.target.value)}
            onKeyDown={(e) => e.key === 'Enter' && handleSend()}
            placeholder="Ask anything..."
            className="w-full bg-slate-900 border border-slate-700 text-slate-100 rounded-xl py-4 pl-4 pr-32 focus:outline-none focus:border-amber-500 focus:ring-1 focus:ring-amber-500 transition-all shadow-inner"
          />
          <div className="absolute right-2 flex gap-2">
            <button className="p-2 text-slate-500 hover:text-slate-300 transition-colors">
              <Paperclip className="w-5 h-5" />
            </button>
            <button
              onClick={handleSend}
              disabled={isLoading || !input.trim()}
              className="p-2 bg-amber-500 hover:bg-amber-400 text-slate-950 rounded-lg disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
            >
              <Send className="w-5 h-5" />
            </button>
          </div>
        </div>
        <div className="mt-2 flex items-center justify-center gap-2 text-xs text-slate-500">
          <Zap size={12} /> Powered by Gemini 2.5 & 3.0 Pro
        </div>
      </div>
    </div>
  );
};

const ToggleBtn = ({ active, onClick, icon: Icon, label, color }: any) => (
  <button
    onClick={onClick}
    className={`flex items-center gap-2 px-3 py-1.5 rounded-lg text-xs font-medium border transition-all ${active
      ? `bg-slate-800 border-slate-600 ${color}`
      : 'bg-transparent border-transparent text-slate-500 hover:bg-slate-800'
      }`}
  >
    <Icon size={14} />
    <span className="hidden md:inline">{label}</span>
    {active && <div className={`w-1.5 h-1.5 rounded-full ${color.replace('text', 'bg')}`} />}
  </button>
);

export default ChatInterface;
