import React, { useState, useEffect } from 'react';
import { Download, Check } from 'lucide-react';
import { authService } from '../services/authService';

interface AutoDownloadIndicatorProps {
    className?: string;
}

export const AutoDownloadIndicator: React.FC<AutoDownloadIndicatorProps> = ({ className = '' }) => {
    const [isEnabled, setIsEnabled] = useState(false);
    const [isLoading, setIsLoading] = useState(true);

    useEffect(() => {
        const checkAutoDownload = async () => {
            try {
                const currentUser = authService.getCurrentUser();
                if (!currentUser) {
                    setIsEnabled(false);
                    setIsLoading(false);
                    return;
                }

                const response = await fetch(`http://localhost:3001/api/user/preferences?uid=${currentUser.uid}`, {
                    headers: authService.getHeaders()
                });

                if (response.ok) {
                    const data = await response.json();
                    setIsEnabled(data?.data?.uiConfig?.autoDownload === true);
                }
            } catch (error) {
                console.error('Failed to check auto-download status:', error);
            } finally {
                setIsLoading(false);
            }
        };

        checkAutoDownload();

        // Re-check when window gains focus (in case settings changed)
        const handleFocus = () => checkAutoDownload();
        window.addEventListener('focus', handleFocus);
        return () => window.removeEventListener('focus', handleFocus);
    }, []);

    if (isLoading || !isEnabled) {
        return null;
    }

    return (
        <div
            className={`flex items-center gap-2 px-3 py-1.5 bg-green-500/10 border border-green-500/30 rounded-lg ${className}`}
            title="Auto-Download is enabled"
        >
            <div className="relative">
                <Download size={14} className="text-green-600 dark:text-green-400" />
                <div className="absolute -top-1 -right-1 w-2.5 h-2.5 bg-green-500 rounded-full flex items-center justify-center">
                    <Check size={8} className="text-white" strokeWidth={3} />
                </div>
            </div>
            <span className="text-[10px] font-bold text-green-600 dark:text-green-400 uppercase tracking-wide">
                Auto-Download
            </span>
        </div>
    );
};
