

import React, { useState, useEffect, useRef, useMemo } from 'react';
import {
    Users, Server, Activity, DollarSign,
    ShieldAlert, Search, Filter, MoreVertical,
    ArrowUpRight, ArrowDownRight, Clock, Database,
    Cpu, HardDrive, Globe, AlertTriangle, CheckCircle2,
    Mail, Inbox, Sliders, BarChart3, FileText, Trash2,
    Eye, X, Send, RotateCcw, Shield, Bell, Lock, UserPlus,
    Megaphone, Calendar, Copy, Zap, PieChart as PieChartIcon,
    Map, AlertOctagon, Download, RefreshCw, Layers, Terminal,
    Key, MousePointer, LogIn, Settings, Bug, Edit, Save, LayoutTemplate, Loader2, CreditCard,
    MessageSquare, ShieldCheck, Archive, AlertCircle, File, TrendingUp, Image as ImageIcon, Video,
    Facebook, Twitter, Heart, User as UserIcon, Paperclip, Lightbulb, Plus, HelpCircle, ExternalLink,
    Info, GraduationCap, GripVertical, Undo2, BrainCircuit, EyeOff, History, ChevronDown,
    Volume2, RefreshCcw, Laptop, Palette, Camera, LayoutDashboard, Sparkles, PencilRuler, Folder, Star, ScanEye, ShieldOff // Added missing icons
} from 'lucide-react';
import {
    AreaChart, Area, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer,
    BarChart, Bar, Legend, PieChart, Pie, Cell, LineChart, Line
} from 'recharts';
import { AdminSection, User, SiteSettings, SystemEmail, Broadcast, BugReport, AuditLog, SupportMessage, Tip, TutorialStep, StockSite, SidebarItemSetting } from '../types';
import { MODELS, MODEL_TYPE_TO_SETTINGS } from "../constants";
import { adminService } from '../services/adminService';

const USAGE_DATA = [
    { name: '00:00', cpu: 12, mem: 24, api: 150 },
    { name: '04:00', cpu: 15, mem: 25, api: 120 },
    { name: '08:00', cpu: 45, mem: 40, api: 850 },
    { name: '12:00', cpu: 68, mem: 55, api: 1240 },
    { name: '16:00', cpu: 55, mem: 48, api: 980 },
    { name: '20:00', cpu: 35, mem: 35, api: 450 },
    { name: '23:59', cpu: 20, mem: 28, api: 200 },
];

// Analytics Mock Data
const TRAFFIC_DATA = [
    { name: 'Mon', users: 4000, sessions: 2400 },
    { name: 'Tue', users: 3000, sessions: 1398 },
    { name: 'Wed', users: 2000, sessions: 9800 },
    { name: 'Thu', users: 2780, sessions: 3908 },
    { name: 'Fri', users: 1890, sessions: 4800 },
    { name: 'Sat', users: 2390, sessions: 3800 },
    { name: 'Sun', users: 3490, sessions: 4300 },
];

const MODEL_USAGE_DATA = [
    { name: 'Gemini Flash', value: 45, color: '#10b981' }, // green
    { name: 'Gemini Pro', value: 25, color: '#8b5cf6' }, // purple
    { name: 'Veo Video', value: 20, color: '#3b82f6' }, // blue
    { name: 'Imagen 4.0', value: 10, color: '#f59e0b' }, // amber
];

const REVENUE_DATA = [
    { name: 'Jan', revenue: 12500 },
    { name: 'Feb', revenue: 15000 },
    { name: 'Mar', revenue: 18200 },
    { name: 'Apr', revenue: 22400 },
    { name: 'May', revenue: 26800 },
    { name: 'Jun', revenue: 31500 },
];

const GEO_DATA = [
    { country: 'United States', users: '45%', trend: '+12%' },
    { country: 'United Kingdom', users: '15%', trend: '+5%' },
    { country: 'Germany', users: '12%', trend: '+8%' },
    { country: 'Japan', users: '8%', trend: '-2%' },
    { country: 'Canada', users: '6%', trend: '+4%' },
];

// ========== Helper Components (must be defined before use) ==========

const StatCard = ({ title, value, trend, icon: Icon, color }: any) => {
    const colors: any = {
        blue: 'text-blue-500 bg-blue-500/10 border-blue-500/20',
        green: 'text-green-500 bg-green-500/10 border-green-500/20',
        purple: 'text-purple-500 bg-purple-500/10 border-purple-500/20',
        amber: 'text-amber-500 bg-amber-500/10 border-amber-500/20',
        red: 'text-red-500 bg-red-500/10 border-red-500/20',
        orange: 'text-orange-500 bg-orange-500/10 border-orange-500/20',
    };

    const theme = colors[color] || colors['blue'];

    return (
        <div className={`p-6 rounded-2xl border ${theme.split(' ')[2]} bg-white dark:bg-slate-900/50 shadow-sm dark:shadow-none`}>
            <div className="flex justify-between items-start mb-4">
                <div className={`p-3 rounded-xl ${theme.split(' ')[1]} `}>
                    <Icon className={theme.split(' ')[0]} size={24} />
                </div>
                <span className={`flex items-center gap-1 text-xs font-bold px-2 py-1 rounded-lg ${trend.startsWith('+') ? 'text-green-600 dark:text-green-400 bg-green-100 dark:bg-green-400/10' : 'text-red-600 dark:text-red-400 bg-red-100 dark:bg-red-400/10'} `}>
                    {trend.startsWith('+') ? <ArrowUpRight size={12} /> : <ArrowDownRight size={12} />}
                    {trend}
                </span>
            </div>
            <h3 className="text-slate-500 dark:text-slate-400 text-sm font-medium">{title}</h3>
            <p className="text-3xl font-bold text-slate-900 dark:text-white font-['Space_Grotesk']">{value}</p>
        </div>
    );
};

const Toggle = ({ checked, onChange }: { checked: boolean, onChange: () => void }) => (
    <button onClick={onChange} className={`w-12 h-6 rounded-full transition-colors relative ${checked ? 'bg-green-500' : 'bg-slate-200 dark:bg-slate-700'} `}>
        <div className={`absolute top-1 left-1 w-4 h-4 bg-white rounded-full transition-transform ${checked ? 'translate-x-6' : 'translate-x-0'} `} />
    </button>
);

const DoubleConfirmDeleteButton = ({ onDelete, className = "", iconSize = 16, title = "Delete" }: { onDelete: () => void, className?: string, iconSize?: number, title?: string }) => {
    const [confirming, setConfirming] = useState(false);
    const [countdown, setCountdown] = useState(3);

    useEffect(() => {
        let timer: NodeJS.Timeout;
        if (confirming) {
            setCountdown(3);
            timer = setInterval(() => {
                setCountdown(prev => {
                    if (prev <= 1) {
                        clearInterval(timer);
                        setConfirming(false);
                        return 3;
                    }
                    return prev - 1;
                });
            }, 1000);
        }
        return () => clearInterval(timer);
    }, [confirming]);

    const handleClick = (e: React.MouseEvent) => {
        e.stopPropagation();
        if (confirming) {
            onDelete();
            setConfirming(false);
        } else {
            setConfirming(true);
        }
    };

    return (
        <button
            onClick={handleClick}
            className={`transition-all duration-200 flex items-center justify-center gap-1 ${className} ${confirming ? 'bg-red-600 text-white hover:bg-red-700 w-auto px-3' : 'hover:bg-red-100 dark:hover:bg-red-900/30 text-slate-400 hover:text-red-600 dark:text-slate-500 dark:hover:text-red-400'} `}
            title={confirming ? "Click again to confirm" : title}
        >
            <Trash2 size={iconSize} />
            {confirming && <span className="text-xs font-bold whitespace-nowrap">Confirm ({countdown})</span>}
        </button>
    );
};

const UserProfileModal = ({ user, onClose }: { user: User, onClose: () => void }) => (
    <div className="fixed inset-0 z-50 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4" onClick={onClose}>
        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl w-full max-w-md overflow-hidden shadow-2xl relative animate-fade-in-up" onClick={e => e.stopPropagation()}>
            <button onClick={onClose} className="absolute top-4 right-4 p-2 bg-slate-100 dark:bg-slate-800 rounded-full text-slate-600 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white transition">
                <X size={18} />
            </button>
            <div className="p-6 text-center">
                <div className="w-20 h-20 rounded-full bg-gradient-to-tr from-blue-500 to-purple-600 flex items-center justify-center text-3xl font-bold text-white mx-auto mb-4 shadow-lg">
                    {user.photoURL ? <img src={user.photoURL} className="w-full h-full rounded-full object-cover" /> : user.displayName[0]}
                </div>
                <h2 className="text-xl font-bold text-slate-900 dark:text-white">{user.displayName}</h2>
                <p className="text-sm text-slate-500">{user.email}</p>
                <div className="mt-2 flex items-center justify-center gap-2">
                    <span className="px-2 py-1 bg-slate-100 dark:bg-slate-800 rounded text-xs font-bold uppercase text-slate-600 dark:text-slate-400">{user.role}</span>
                    <span className="px-2 py-1 bg-amber-100 dark:bg-amber-900/30 rounded text-xs font-bold uppercase text-amber-600 dark:text-amber-400">{user.subscription} Plan</span>
                </div>
                <button
                    onClick={() => window.open(`/?admin_view_as=${user.uid}`, '_blank')}
                    className="mt-4 px-4 py-2 bg-slate-900 dark:bg-white text-white dark:text-slate-900 text-sm font-bold rounded-lg flex items-center justify-center gap-2 mx-auto hover:opacity-90 transition"
                >
                    <LayoutDashboard size={14} /> View User Dashboard
                </button>
                <div className="grid grid-cols-2 gap-4 mt-6 text-left">
                    <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800"><p className="text-[10px] text-slate-500 uppercase font-bold">Joined</p><p className="text-sm font-bold text-slate-800 dark:text-slate-200">{new Date(user.createdAt).toLocaleDateString()}</p></div>
                    <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800"><p className="text-[10px] text-slate-500 uppercase font-bold">Last Login</p><p className="text-sm font-bold text-slate-800 dark:text-slate-200">{new Date(user.lastLoginAt).toLocaleDateString()}</p></div>
                    <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800"><p className="text-[10px] text-slate-500 uppercase font-bold">Images Gen</p><p className="text-sm font-bold text-slate-800 dark:text-slate-200">{user.usageStats.imagesGenerated}</p></div>
                    <div className="p-3 bg-slate-50 dark:bg-slate-950 rounded-xl border border-slate-200 dark:border-slate-800"><p className="text-[10px] text-slate-500 uppercase font-bold">Status</p><p className={`text-sm font-bold ${user.isBanned ? 'text-red-500' : 'text-green-500'} `}>{user.isBanned ? 'Banned' : 'Active'}</p></div>
                </div>
            </div>
        </div>
    </div>
);

// ========== End Helper Components ==========

interface AdminDashboardProps {
    adminSection: AdminSection;
}

const AdminDashboard: React.FC<AdminDashboardProps> = ({ adminSection }) => {
    const [selectedMessageUserId, setSelectedMessageUserId] = useState<string | null>(null);
    // We need to force the tab to switch when a user is selected for messaging
    // However, the active tab is controlled by the parent (App.tsx) via adminSection prop.
    // This means we can't easily switch tabs internally if the parent controls it.
    // BUT, looking at the code, AdminDashboard seems to be the one rendering the content based on props.
    // If we want to switch to MESSAGES tab, we might need to notify the parent or handle it differently.

    // Actually, looking at the previous code I saw in the outline, AdminDashboard takes 'adminSection' as a prop.
    // This means the Sidebar/Navigation is likely OUTSIDE this component (in App.tsx).
    // If so, we can't just 'setActiveTab' here.

    // Let's check how App.tsx uses AdminDashboard.
    // If I can't change the tab from here, I might need to dispatch an event or use a callback.

    // For now, let's assume we can just render the MessagesView with the selected user if we are in the Messages section.
    // But wait, the user clicks "Message" in "Users" section. We need to switch to "Messages" section.

    // If the parent controls the section, we need a way to request a section change.
    // Let's check if there's a dispatch or something.

    // Alternative: We can just pass the selectedUser to the MessagesView when it IS rendered.
    // But we need to switch to it.

    // Let's look at how to switch tabs.
    // If I can't switch tabs, I'll just show a modal or something? No, that's messy.

    // Let's try to dispatch a custom event to request navigation.
    const handleMessageUser = (uid: string) => {
        console.log("AdminDashboard handling message user:", uid);
        setSelectedMessageUserId(uid);
        window.dispatchEvent(new CustomEvent('sf_navigate_admin', { detail: 'MESSAGES' }));
    };

    const renderContent = () => {
        switch (adminSection) {
            case AdminSection.USERS: return <UsersManagement onMessageUser={handleMessageUser} />;
            case AdminSection.SITE_SETTINGS: return <SiteConfiguration />;
            case AdminSection.AUTH_SETTINGS: return <AuthConfiguration />;
            case AdminSection.LANDING_PAGE: return <LandingPageEditor />;
            case AdminSection.BUG_REPORTS: return <BugReports />;
            case AdminSection.MESSAGES: return <MessagesView selectedUserId={selectedMessageUserId} setSelectedUserId={setSelectedMessageUserId} />;
            case AdminSection.BROADCASTS: return <BroadcastsView />;
            case AdminSection.INBOX: return <InboxView />;
            case AdminSection.EMAIL_SETTINGS: return <EmailSettingsView />;
            case AdminSection.AUDIT: return <AuditLogView />;
            case AdminSection.TIPS: return <TipsManagement />;
            case AdminSection.TUTORIAL_STEPS: return <TutorialStepsManagement />;
            case AdminSection.MARKET_INTELLIGENCE: return <MarketIntelligenceAdmin />;
            case AdminSection.TRASH: return <AdminRecycleBin />;
            default: return <Overview />;
        }
    };

    return (
        <div className="p-6 lg:p-10 max-w-[1600px] mx-auto space-y-8 animate-fade-in text-slate-900 dark:text-slate-100">
            {renderContent()}
        </div>
    );
};

// ... (Existing Components)

const Overview = () => {
    const [stats, setStats] = useState({ users: 0, totalGeneratedFiles: 0, totalApiCalls: 0, totalPromptsGenerated: 0 });
    const [loading, setLoading] = useState(true);
    const [analyticsData, setAnalyticsData] = useState<any>(null);
    const [resetClickCount, setResetClickCount] = useState(0);
    const [resetCountdown, setResetCountdown] = useState<number | null>(null);
    const [resetting, setResetting] = useState(false);

    useEffect(() => {
        const fetchStats = async () => {
            try {
                console.log('[Frontend-Overview] Fetching admin stats...');
                const data = await adminService.getAnalytics();
                console.log('[Frontend-Overview] Received analytics:', data);

                setStats({
                    users: data.users || 0,
                    totalGeneratedFiles: data.totalGeneratedFiles || 0,
                    totalApiCalls: data.totalApiCalls || 0,
                    totalPromptsGenerated: data.totalPromptsGenerated || 0
                });

                setAnalyticsData(data);
            } catch (error) {
                console.error('[Frontend-Overview] Failed to fetch admin stats:', error);
            } finally {
                setLoading(false);
            }
        };

        fetchStats();
        // Refresh every 30 seconds
        const interval = setInterval(fetchStats, 30000);
        return () => clearInterval(interval);
    }, []);

    // Reset stats double-click logic
    useEffect(() => {
        if (resetClickCount === 1) {
            setResetCountdown(3);
            const timer = setInterval(() => {
                setResetCountdown(prev => {
                    if (prev === null || prev <= 1) {
                        setResetClickCount(0);
                        return null;
                    }
                    return prev - 1;
                });
            }, 1000);

            return () => clearInterval(timer);
        }
    }, [resetClickCount]);

    const handleResetStats = async () => {
        if (resetClickCount === 0) {
            // First click
            setResetClickCount(1);
        } else if (resetClickCount === 1 && resetCountdown !== null) {
            // Second click within 3 seconds
            setResetClickCount(0);
            setResetCountdown(null);
            setResetting(true);

            try {
                await adminService.resetStats();
                // Refresh stats
                window.location.reload();
            } catch (error: any) {
                console.error('Failed to reset stats:', error);
            } finally {
                setResetting(false);
            }
        }
    };

    const formatNumber = (num: number) => {
        if (num >= 1000000) return (num / 1000000).toFixed(1) + 'M';
        if (num >= 1000) return (num / 1000).toFixed(1) + 'K';
        return num.toString();
    };

    return (
        <div className="space-y-8">
            <div className="flex items-center justify-between">
                <h1 className="text-3xl font-bold text-slate-900 dark:text-slate-100">System Overview</h1>
                <button
                    onClick={handleResetStats}
                    disabled={resetting}
                    className={`
                        px-4 py-2 rounded-lg font-medium text-sm transition-all flex items-center gap-2
                        ${resetClickCount === 1 && resetCountdown !== null
                            ? 'bg-red-600 text-white hover:bg-red-700 animate-pulse shadow-lg shadow-red-500/50'
                            : 'bg-slate-200 dark:bg-slate-800 text-slate-700 dark:text-slate-300 hover:bg-red-100 dark:hover:bg-red-900/30 hover:text-red-600 dark:hover:text-red-400'
                        }
                        ${resetting ? 'opacity-50 cursor-not-allowed' : 'cursor-pointer'}
                    `}
                >
                    {resetting ? (
                        <>
                            <Loader2 className="animate-spin" size={16} />
                            Resetting...
                        </>
                    ) : resetClickCount === 1 && resetCountdown !== null ? (
                        <>
                            <AlertCircle size={16} />
                            Click again to confirm ({resetCountdown}s)
                        </>
                    ) : (
                        <>
                            <Trash2 size={16} />
                            Reset All Stats
                        </>
                    )}
                </button>
            </div>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <StatCard
                    title="Total Users"
                    value={loading ? "..." : stats.users.toLocaleString()}
                    trend=""
                    icon={Users}
                    color="blue"
                />
                <StatCard
                    title="Total Generated Files"
                    value={loading ? "..." : stats.totalGeneratedFiles.toLocaleString()}
                    trend=""
                    icon={Activity}
                    color="green"
                />
                <StatCard
                    title="API Requests"
                    value={loading ? "..." : formatNumber(stats.totalApiCalls)}
                    trend=""
                    icon={Server}
                    color="purple"
                />
                <StatCard
                    title="Prompts Generated"
                    value={loading ? "..." : stats.totalPromptsGenerated.toLocaleString()}
                    trend=""
                    icon={Sparkles}
                    color="orange"
                />
            </div>

            {/* Analytics Dashboard Section */}
            <div className="bg-slate-50 dark:bg-slate-900/30 border border-slate-200 dark:border-slate-800 rounded-2xl p-6">
                <h2 className="text-2xl font-bold text-slate-900 dark:text-slate-100 mb-6">Analytics Dashboard</h2>

                {loading ? (
                    <div className="flex items-center justify-center h-64">
                        <Loader2 className="animate-spin text-blue-500" size={48} />
                    </div>
                ) : (
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                        {/* Weekly Traffic */}
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-6">
                            <h3 className="text-lg font-bold text-slate-900 dark:text-slate-200 mb-4">Weekly Traffic</h3>
                            {analyticsData?.weeklyTraffic && analyticsData.weeklyTraffic.length > 0 ? (
                                <>
                                    <ResponsiveContainer width="100%" height={250}>
                                        <BarChart data={analyticsData.weeklyTraffic}>
                                            <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.3} vertical={false} />
                                            <XAxis dataKey="name" stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                            <YAxis stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                            <Tooltip contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc' }} />
                                            <Bar dataKey="users" fill="#3b82f6" radius={[8, 8, 0, 0]} />
                                            <Bar dataKey="sessions" fill="#8b5cf6" radius={[8, 8, 0, 0]} />
                                        </BarChart>
                                    </ResponsiveContainer>
                                    <div className="flex justify-center gap-4 mt-4 text-xs">
                                        <div className="flex items-center gap-2">
                                            <div className="w-3 h-3 rounded-full bg-blue-500"></div>
                                            <span className="text-slate-600 dark:text-slate-400 font-medium">Users</span>
                                        </div>
                                        <div className="flex items-center gap-2">
                                            <div className="w-3 h-3 rounded-full bg-purple-500"></div>
                                            <span className="text-slate-600 dark:text-slate-400 font-medium">Sessions</span>
                                        </div>
                                    </div>
                                </>
                            ) : (
                                <div className="flex items-center justify-center h-64 text-slate-500">No traffic data</div>
                            )}
                        </div>

                        {/* Model Usage Distribution */}
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-6">
                            <h3 className="text-lg font-bold text-slate-900 dark:text-slate-200 mb-4">Model Usage Distribution</h3>
                            {analyticsData?.modelUsageDistribution && analyticsData.modelUsageDistribution.length > 0 ? (
                                <>
                                    <ResponsiveContainer width="100%" height={250}>
                                        <PieChart>
                                            <Pie
                                                data={analyticsData.modelUsageDistribution}
                                                cx="50%"
                                                cy="50%"
                                                innerRadius={60}
                                                outerRadius={90}
                                                paddingAngle={2}
                                                dataKey="value"
                                            >
                                                {analyticsData.modelUsageDistribution.map((entry: any, index: number) => (
                                                    <Cell key={`cell-${index}`} fill={entry.color} />
                                                ))}
                                            </Pie>
                                            <Tooltip contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc' }} />
                                        </PieChart>
                                    </ResponsiveContainer>
                                    <div className="grid grid-cols-2 gap-2 mt-4 text-xs">
                                        {analyticsData.modelUsageDistribution.map((item: any, i: number) => (
                                            <div key={i} className="flex items-center gap-2">
                                                <div className="w-3 h-3 rounded-full" style={{ backgroundColor: item.color }}></div>
                                                <span className="text-slate-600 dark:text-slate-400 font-medium">{item.name}</span>
                                                <span className="text-slate-900 dark:text-slate-200 font-bold ml-auto">{item.value}</span>
                                            </div>
                                        ))}
                                    </div>
                                </>
                            ) : (
                                <div className="flex items-center justify-center h-64 text-slate-500">No model usage data</div>
                            )}
                        </div>

                        {/* Monthly Revenue */}
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-6">
                            <h3 className="text-lg font-bold text-slate-900 dark:text-slate-200 mb-4">Monthly Revenue</h3>
                            {analyticsData?.monthlyRevenue && analyticsData.monthlyRevenue.length > 0 ? (
                                <ResponsiveContainer width="100%" height={250}>
                                    <LineChart data={analyticsData.monthlyRevenue}>
                                        <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.3} vertical={false} />
                                        <XAxis dataKey="name" stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                        <YAxis stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                        <Tooltip contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc' }} />
                                        <Line type="monotone" dataKey="revenue" stroke="#f59e0b" strokeWidth={3} dot={{ fill: '#f59e0b', r: 5 }} />
                                    </LineChart>
                                </ResponsiveContainer>
                            ) : (
                                <div className="flex items-center justify-center h-64 text-slate-500">No revenue data</div>
                            )}
                        </div>

                        {/* Top Countries */}
                        <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-6">
                            <h3 className="text-lg font-bold text-slate-900 dark:text-slate-200 mb-4">Top Countries</h3>
                            {analyticsData?.topCountries && analyticsData.topCountries.length > 0 ? (
                                <div className="space-y-3">
                                    {analyticsData.topCountries.map((country: any, i: number) => (
                                        <div key={i} className="flex items-center justify-between py-2 border-b border-slate-100 dark:border-slate-800 last:border-0">
                                            <span className="text-sm text-slate-700 dark:text-slate-300 font-medium">{country.country}</span>
                                            <div className="flex items-center gap-3">
                                                <span className="text-sm font-bold text-slate-900 dark:text-slate-100">{country.users}</span>
                                                <span className={`text-xs font-bold ${country.trend.startsWith('+') ? 'text-green-600' : 'text-red-600'}`}>
                                                    {country.trend}
                                                </span>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            ) : (
                                <div className="flex items-center justify-center h-64 text-slate-500">No geographic data</div>
                            )}
                        </div>
                    </div>
                )}
            </div>

            {/* System Load & API Usage */}
            <div className="h-96 bg-white dark:bg-slate-900/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm dark:shadow-none">
                <h3 className="text-lg font-bold text-slate-900 dark:text-slate-200 mb-4">System Load & API Usage</h3>
                {loading ? (
                    <div className="flex items-center justify-center h-full">
                        <Loader2 className="animate-spin text-blue-500" size={32} />
                    </div>
                ) : analyticsData?.systemLoadData && analyticsData.systemLoadData.length > 0 ? (
                    <ResponsiveContainer width="100%" height="100%">
                        <AreaChart data={analyticsData.systemLoadData}>
                            <defs>
                                <linearGradient id="colorApi" x1="0" y1="0" x2="0" y2="1">
                                    <stop offset="5%" stopColor="#8b5cf6" stopOpacity={0.3} />
                                    <stop offset="95%" stopColor="#8b5cf6" stopOpacity={0} />
                                </linearGradient>
                            </defs>
                            <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.3} vertical={false} />
                            <XAxis dataKey="name" stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                            <YAxis stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                            <Tooltip contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc' }} itemStyle={{ color: '#e2e8f0' }} />
                            <Area type="monotone" dataKey="api" stroke="#8b5cf6" fillOpacity={1} fill="url(#colorApi)" strokeWidth={3} />
                        </AreaChart>
                    </ResponsiveContainer>
                ) : (
                    <div className="flex items-center justify-center h-full text-slate-500">No system load data</div>
                )}
            </div>
        </div>
    );
};

const MarketIntelligenceAdmin = () => {
    const [sites, setSites] = useState<StockSite[]>([]);
    const [loading, setLoading] = useState(true);
    const [search, setSearch] = useState('');
    const [filterCategory, setFilterCategory] = useState('all');
    const [filterVisibility, setFilterVisibility] = useState('all');
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editingSite, setEditingSite] = useState<StockSite | null>(null);
    const [formData, setFormData] = useState<Partial<StockSite>>({});

    useEffect(() => {
        loadSites();
    }, []);

    const loadSites = async () => {
        const data = await adminService.getMicrostockSites();
        setSites(data.filter(s => !s.isDeleted));
    };

    const handleToggle = async (id: string, field: 'reviewer' | 'market') => {
        // Optimistic update
        setSites(prev => prev.map(s => {
            if (s.id !== id) return s;
            if (field === 'market') return { ...s, isVisibleMarket: !s.isVisibleMarket };
            return { ...s, isVisible: !s.isVisible };
        }));
        await adminService.toggleMicrostockSite(id, field);
        loadSites(); // authoritative sync
    };

    const handleDelete = async (id: string) => {
        try {
            await adminService.deleteMicrostockSite(id);
            await loadSites();
        } catch (error) {
            console.error("Delete failed", error);
            alert("Failed to delete site");
        }
    };

    const openModal = (site?: StockSite) => {
        if (site) {
            setEditingSite(site);
            setFormData({ ...site });
        } else {
            setEditingSite(null);
            setFormData({
                name: '', url: '', category: 'Major', description: '',
                contributors: '', librarySize: '', commission: '',
                isAiPowered: false, isVisible: true, pros: [], cons: []
            });
        }
        setIsModalOpen(true);
    };

    const handleSave = async () => {
        if (!formData.name || !formData.url) return alert("Name and URL are required");

        const siteToSave: any = {
            ...formData,
            id: editingSite ? editingSite.id : undefined
        };

        await adminService.saveMicrostockSite(siteToSave);
        setIsModalOpen(false);
        loadSites();
    };

    const handleArrayInput = (field: 'pros' | 'cons', value: string) => {
        setFormData(prev => ({ ...prev, [field]: value.split('\n').filter(line => line.trim()) }));
    };

    const filteredSites = sites.filter(site => {
        const matchesSearch = site.name.toLowerCase().includes(search.toLowerCase()) ||
            site.url.toLowerCase().includes(search.toLowerCase());
        const matchesCategory = filterCategory === 'all' || site.category === filterCategory;
        const matchesVisibility = filterVisibility === 'all' ||
            (filterVisibility === 'active' && (site.isVisible || site.isVisibleMarket)) ||
            (filterVisibility === 'hidden' && (!site.isVisible && !site.isVisibleMarket));

        return matchesSearch && matchesCategory && matchesVisibility;
    });

    const handlePopulate = async () => {
        await adminService.populateDefaultSites();
        loadSites();
    };

    const handleToggleAll = async (field: 'reviewer' | 'market') => {
        // Determine new state: if any are hidden, unhide all. Else, hide all.
        const currentSites = sites.filter(s => !s.isDeleted);
        const allVisible = currentSites.every(s => field === 'market' ? s.isVisibleMarket : s.isVisible);
        const targetState = !allVisible;

        // Optimistic update
        setSites(prev => prev.map(s => {
            if (field === 'market') return { ...s, isVisibleMarket: targetState };
            return { ...s, isVisible: targetState };
        }));

        await adminService.setAllMicrostockVisibility(field, targetState);
        loadSites();
    };

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex justify-between items-center">
                <div className="flex items-center gap-4">
                    <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2">
                        <Globe className="text-blue-500" /> Market Intelligence Hub
                    </h2>
                    <span className="px-3 py-1 bg-slate-100 dark:bg-slate-800 rounded-full text-sm font-medium text-slate-500">
                        {filteredSites.length} Sites
                    </span>
                </div>
                <div className="flex gap-4">
                    <select
                        value={filterCategory}
                        onChange={(e) => setFilterCategory(e.target.value)}
                        className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl px-4 py-2 text-sm focus:ring-2 focus:ring-blue-500 outline-none transition"
                    >
                        <option value="all">All Categories</option>
                        <option value="Major">Major</option>
                        <option value="Mid-Tier">Mid-Tier</option>
                        <option value="Video">Video</option>
                    </select>
                    <select
                        value={filterVisibility}
                        onChange={(e) => setFilterVisibility(e.target.value)}
                        className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl px-4 py-2 text-sm focus:ring-2 focus:ring-blue-500 outline-none transition"
                    >
                        <option value="all">All Status</option>
                        <option value="active">Active</option>
                        <option value="hidden">Hidden</option>
                    </select>
                    <button onClick={handlePopulate} className="px-4 py-2 bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 font-bold rounded-xl hover:bg-slate-200 dark:hover:bg-slate-700 transition flex items-center gap-2">
                        <Database size={18} /> Populate Defaults
                    </button>
                    <button onClick={() => openModal()} className="px-4 py-2 bg-amber-500 text-slate-950 font-bold rounded-xl hover:bg-amber-600 transition flex items-center gap-2">
                        <Plus size={18} /> Add Site
                    </button>
                </div>
            </div>

            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden shadow-sm">
                <div className="max-h-[700px] overflow-y-auto custom-scrollbar">
                    <table className="w-full text-sm text-left">
                        <thead className="bg-slate-50 dark:bg-slate-950 text-slate-500 border-b border-slate-200 dark:border-slate-800 sticky top-0 z-10 transition-colors">
                            <tr>
                                <th className="p-4 font-bold">Platform Name</th>
                                <th className="p-4 font-bold">Category</th>
                                <th className="p-4 font-bold text-center group cursor-pointer hover:bg-slate-100 dark:hover:bg-slate-900 transition-colors" title="Toggle All Market Visibility" onClick={() => handleToggleAll('market')}>
                                    <div className="flex items-center justify-center gap-2">
                                        <Globe size={16} />
                                        <span className="text-[10px] uppercase font-bold text-blue-500 whitespace-nowrap hidden lg:inline-block">Toggle All</span>
                                    </div>
                                </th>
                                <th className="p-4 font-bold text-center group cursor-pointer hover:bg-slate-100 dark:hover:bg-slate-900 transition-colors" title="Toggle All Reviewer Visibility" onClick={() => handleToggleAll('reviewer')}>
                                    <div className="flex items-center justify-center gap-2">
                                        <Eye size={16} />
                                        <span className="text-[10px] uppercase font-bold text-green-500 whitespace-nowrap hidden lg:inline-block">Toggle All</span>
                                    </div>
                                </th>
                                <th className="p-4 font-bold text-right">Actions</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-slate-100 dark:divide-slate-800">
                            {filteredSites.map(site => (
                                <tr key={site.id} className="hover:bg-slate-50 dark:hover:bg-slate-800/50 transition">
                                    <td className="p-4">
                                        <div className="flex items-center gap-3">
                                            <div className="w-8 h-8 rounded-lg bg-slate-100 dark:bg-slate-800 flex items-center justify-center font-bold text-slate-500">
                                                {site.name.charAt(0)}
                                            </div>
                                            <div>
                                                <div className="font-bold text-slate-900 dark:text-white flex items-center gap-2">
                                                    {site.name}
                                                    {site.isAiPowered && <span className="px-1.5 py-0.5 bg-purple-100 text-purple-600 dark:bg-purple-900/30 dark:text-purple-400 text-[10px] rounded uppercase font-bold">AI</span>}
                                                </div>
                                                <a href={site.url} target="_blank" rel="noreferrer" className="text-xs text-slate-400 hover:text-blue-500 hover:underline">{site.url}</a>
                                            </div>
                                        </div>
                                    </td>
                                    <td className="p-4">
                                        <span className="px-2 py-1 bg-slate-100 dark:bg-slate-800 rounded text-xs font-bold text-slate-600 dark:text-slate-400">{site.category}</span>
                                    </td>
                                    {/* Market Visibility */}
                                    <td className="p-4 text-center">
                                        <button onClick={() => handleToggle(site.id, 'market')} className={`p-2 rounded-lg transition ${site.isVisibleMarket ? 'text-blue-500 bg-blue-50 dark:bg-blue-900/20' : 'text-red-500 bg-red-50 dark:bg-red-900/20'}`} title={site.isVisibleMarket ? "Hide from Market Intel" : "Show in Market Intel"}>
                                            <Globe size={16} />
                                        </button>
                                    </td>
                                    {/* Reviewer Visibility */}
                                    <td className="p-4 text-center">
                                        <button onClick={() => handleToggle(site.id, 'reviewer')} className={`p-2 rounded-lg transition ${site.isVisible ? 'text-green-500 bg-green-50 dark:bg-green-900/20' : 'text-red-500 bg-red-50 dark:bg-red-900/20'}`} title={site.isVisible ? "Hide from Reviewer" : "Show in Reviewer"}>
                                            <Eye size={16} />
                                        </button>
                                    </td>
                                    <td className="p-4 text-right flex justify-end gap-2">
                                        <button onClick={() => openModal(site)} className="p-2 text-blue-500 hover:bg-blue-50 dark:hover:bg-blue-900/20 rounded-lg transition" title="Edit Site">
                                            <Edit size={16} />
                                        </button>
                                        <DoubleConfirmDeleteButton onDelete={() => handleDelete(site.id)} className="p-2 text-red-500 hover:bg-red-50 dark:hover:bg-red-900/20 rounded-lg transition" />
                                    </td>
                                </tr>
                            ))}
                        </tbody>
                    </table>
                </div>
            </div>

            {isModalOpen && (
                <div className="fixed inset-0 z-50 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4" onClick={() => setIsModalOpen(false)}>
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl w-full max-w-2xl p-6 shadow-2xl animate-fade-in-up max-h-[90vh] overflow-y-auto custom-scrollbar" onClick={e => e.stopPropagation()}>
                        <div className="flex justify-between items-center mb-6">
                            <h3 className="text-xl font-bold text-slate-900 dark:text-white">{editingSite ? 'Edit Site' : 'Add New Site'}</h3>
                            <button onClick={() => setIsModalOpen(false)} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-slate-500"><X size={20} /></button>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Name</label><input type="text" value={formData.name} onChange={e => setFormData({ ...formData, name: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500" /></div>
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">URL</label><input type="text" value={formData.url} onChange={e => setFormData({ ...formData, url: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500" /></div>
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1">Category</label>
                                <select value={formData.category} onChange={e => setFormData({ ...formData, category: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500">
                                    <option value="Major">Major</option>
                                    <option value="Video">Video</option>
                                    <option value="Design">Design</option>
                                    <option value="Specialty">Specialty</option>
                                    <option value="Subscription">Subscription</option>
                                </select>
                            </div>
                            <div className="flex items-center gap-4 pt-6">
                                <label className="flex items-center gap-2 cursor-pointer">
                                    <input type="checkbox" checked={formData.isAiPowered || false} onChange={e => setFormData({ ...formData, isAiPowered: e.target.checked })} className="w-4 h-4 rounded border-slate-300 text-amber-500 focus:ring-amber-500" />
                                    <span className="text-sm font-bold text-slate-700 dark:text-slate-300">AI Powered</span>
                                </label>
                                <label className="flex items-center gap-2 cursor-pointer">
                                    <input type="checkbox" checked={formData.isVisible !== false} onChange={e => setFormData({ ...formData, isVisible: e.target.checked })} className="w-4 h-4 rounded border-slate-300 text-amber-500 focus:ring-amber-500" />
                                    <span className="text-sm font-bold text-slate-700 dark:text-slate-300">Visible</span>
                                </label>
                            </div>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Contributors</label><input type="text" value={formData.contributors} onChange={e => setFormData({ ...formData, contributors: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500" placeholder="e.g. 1M+" /></div>
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Library Size</label><input type="text" value={formData.librarySize} onChange={e => setFormData({ ...formData, librarySize: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500" placeholder="e.g. 500M+" /></div>
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Commission</label><input type="text" value={formData.commission} onChange={e => setFormData({ ...formData, commission: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500" placeholder="e.g. 15-40%" /></div>
                        </div>

                        <div className="space-y-4 mb-6">
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Description</label><textarea value={formData.description} onChange={e => setFormData({ ...formData, description: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500 h-20 resize-none" /></div>
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Pros (One per line)</label><textarea value={formData.pros?.join('\n')} onChange={e => handleArrayInput('pros', e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500 h-24 resize-none" placeholder="- High traffic&#10;- Good rates" /></div>
                            <div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Cons (One per line)</label><textarea value={formData.cons?.join('\n')} onChange={e => handleArrayInput('cons', e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500 h-24 resize-none" placeholder="- High competition&#10;- Slow review" /></div>
                        </div>

                        <div className="flex justify-end gap-2 pt-4 border-t border-slate-200 dark:border-slate-800">
                            <button onClick={() => setIsModalOpen(false)} className="px-4 py-2 text-slate-500 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-sm font-bold">Cancel</button>
                            <button onClick={handleSave} className="px-6 py-2 bg-amber-500 hover:bg-amber-600 text-slate-950 rounded-lg text-sm font-bold">Save Site</button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
};


const AdminRecycleBin = () => {
    const [activeTab, setActiveTab] = useState<'bugs' | 'users' | 'tips' | 'steps' | 'broadcasts'>('bugs');
    const [deletedItems, setDeletedItems] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        loadDeletedItems();
    }, [activeTab]);

    const loadDeletedItems = async () => {
        setLoading(true);
        let items: any[] = [];
        if (activeTab === 'bugs') {
            const data = await adminService.getBugReports();
            items = data.filter(i => i.isDeleted);
        } else if (activeTab === 'users') {
            const data = await adminService.getUsers();
            items = data.filter(i => i.isDeleted);
        } else if (activeTab === 'tips') {
            const data = await adminService.getAllTips();
            items = data.filter(i => i.isDeleted);
        } else if (activeTab === 'steps') {
            const data = await adminService.getTutorialSteps();
            items = data.filter(i => i.isDeleted);
        } else if (activeTab === 'broadcasts') {
            const data = await adminService.getBroadcasts();
            items = data.filter(i => i.isDeleted);
        }
        setDeletedItems(items);
        setLoading(false);
    };

    const handleRestore = async (id: string) => {
        if (activeTab === 'bugs') await adminService.restoreBugReport(id);
        else if (activeTab === 'users') await adminService.restoreUser(id);
        else if (activeTab === 'tips') await adminService.restoreTip(id);
        else if (activeTab === 'steps') await adminService.restoreTutorialStep(id);
        else if (activeTab === 'broadcasts') await adminService.restoreBroadcast(id);
        loadDeletedItems();
    };

    const handlePermanentDelete = async (id: string) => {
        try {
            if (activeTab === 'bugs') await adminService.permanentDeleteBugReport(id);
            else if (activeTab === 'users') await adminService.permanentDeleteUser(id);
            else if (activeTab === 'tips') await adminService.permanentDeleteTip(id);
            else if (activeTab === 'steps') await adminService.permanentDeleteTutorialStep(id);
            else if (activeTab === 'broadcasts') await adminService.permanentDeleteBroadcast(id);
            await loadDeletedItems();
        } catch (error) {
            console.error("Delete failed", error);
            alert("Failed to delete item");
        }
    };

    const handleDeleteAll = async () => {
        if (deletedItems.length === 0) return;
        if (deletedItems.length === 0) return;
        // if (!confirm(`Are you sure you want to PERMANENTLY delete ALL ${deletedItems.length} items in ${activeTab}? This cannot be undone.`)) return;

        setLoading(true);
        try {
            await Promise.all(deletedItems.map(item => {
                const id = item.id || item.uid;
                if (activeTab === 'bugs') return adminService.permanentDeleteBugReport(id);
                if (activeTab === 'users') return adminService.permanentDeleteUser(id);
                if (activeTab === 'tips') return adminService.permanentDeleteTip(id);
                if (activeTab === 'steps') return adminService.permanentDeleteTutorialStep(id);
                if (activeTab === 'broadcasts') return adminService.permanentDeleteBroadcast(id);
                return Promise.resolve();
            }));
            await loadDeletedItems();
        } catch (error) {
            console.error("Delete all failed", error);
            alert("Failed to delete all items.");
            setLoading(false);
        }
    };

    return (
        <div className="space-y-6">
            <div className="flex items-center justify-between gap-3">
                <div className="space-y-1">
                    <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2">
                        <Trash2 className="text-red-500" /> Admin Recycle Bin
                    </h2>
                    <p className="text-slate-500 dark:text-slate-400 text-sm">Manage soft-deleted items. Restore or permanently remove them.</p>
                </div>

                {deletedItems.length > 0 && (
                    <button
                        onClick={handleDeleteAll}
                        className="px-4 py-2 bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 hover:bg-red-600 hover:text-white dark:hover:bg-red-600 dark:hover:text-white font-bold rounded-xl transition flex items-center gap-2"
                    >
                        <Trash2 size={18} /> Delete All
                    </button>
                )}
            </div>

            <div className="flex border-b border-slate-200 dark:border-slate-800 overflow-x-auto">
                {['bugs', 'users', 'tips', 'steps', 'broadcasts'].map((tab) => (
                    <button
                        key={tab}
                        onClick={() => setActiveTab(tab as any)}
                        className={`px-6 py-3 text-sm font-bold capitalize transition-colors border-b-2 whitespace-nowrap ${activeTab === tab
                            ? 'border-red-500 text-red-600 dark:text-red-400'
                            : 'border-transparent text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'
                            }`}
                    >
                        {tab === 'steps' ? 'Tutorials' : tab}
                    </button>
                ))}
            </div>

            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden shadow-sm">
                <div className="divide-y divide-slate-100 dark:divide-slate-800 max-h-[600px] overflow-y-auto">
                    {loading ? (
                        <div className="p-8 flex justify-center"><Loader2 className="animate-spin text-slate-400" /></div>
                    ) : deletedItems.length === 0 ? (
                        <div className="p-8 text-center text-slate-500 text-sm italic">Recycle bin is empty for {activeTab}.</div>
                    ) : (
                        deletedItems.map((item: any) => (
                            <div key={item.id || item.uid} className="p-4 flex items-center justify-between hover:bg-slate-50 dark:hover:bg-slate-800/50 transition">
                                <div className="flex-1 min-w-0 pr-4">
                                    <h4 className="text-sm font-bold text-slate-700 dark:text-slate-300 truncate">
                                        {activeTab === 'users' ? item.displayName : (item.title || 'Untitled Item')}
                                    </h4>
                                    <p className="text-xs text-slate-500 truncate">
                                        {activeTab === 'users' ? item.email : (item.description || item.text || item.content || 'No description')}
                                    </p>
                                </div>
                                <div className="flex items-center gap-2 shrink-0">
                                    <button
                                        onClick={() => handleRestore(item.id || item.uid)}
                                        className="p-2 bg-blue-50 dark:bg-blue-900/20 text-blue-600 dark:text-blue-400 rounded-lg hover:bg-blue-100 dark:hover:bg-blue-900/40 transition"
                                        title="Restore"
                                    >
                                        <Undo2 size={16} />
                                    </button>
                                    <DoubleConfirmDeleteButton
                                        onDelete={() => handlePermanentDelete(item.id || item.uid)}
                                        className="p-2 bg-red-50 dark:bg-red-900/20 text-red-600 dark:text-red-400 rounded-lg hover:bg-red-100 dark:hover:bg-red-900/40"
                                        title="Delete Forever"
                                    />
                                </div>
                            </div>
                        ))
                    )}
                </div>
            </div>
        </div>
    );
};

const BugReports = () => {
    const [bugs, setBugs] = useState<BugReport[]>([]);
    const [filter, setFilter] = useState('all'); // all, open, in_progress, resolved

    useEffect(() => { loadBugs(); }, []);

    const loadBugs = async () => {
        const data = await adminService.getBugReports();
        setBugs(data.filter(b => !b.isDeleted));
    };

    const handleStatusChange = async (id: string, status: any) => {
        await adminService.updateBugStatus(id, status);
        loadBugs();
        window.dispatchEvent(new Event('sf_bug_updated'));
    };

    const handleDelete = async (id: string) => {
        // if (confirm('Are you sure you want to delete this report?')) {
        try {
            await adminService.deleteBugReport(id);
            await loadBugs();
            window.dispatchEvent(new Event('sf_bug_updated'));
        } catch (error) {
            console.error("Delete failed", error);
            alert("Failed to delete bug report");
        }
        // }
    };

    const filteredBugs = bugs.filter(b => filter === 'all' ? true : b.status === filter);

    return (
        <div className="space-y-6">
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2">
                    <Bug className="text-red-500" /> Bug Reports
                </h2>
                <div className="flex bg-slate-100 dark:bg-slate-900 p-1 rounded-lg border border-slate-200 dark:border-slate-800">
                    {['all', 'open', 'in_progress', 'resolved'].map(f => (
                        <button
                            key={f}
                            onClick={() => setFilter(f)}
                            className={`px-3 py-1.5 rounded-md text-xs font-bold capitalize transition ${filter === f ? 'bg-white dark:bg-slate-800 shadow-sm text-slate-900 dark:text-white' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                        >
                            {f.replace('_', ' ')}
                        </button>
                    ))}
                </div>
            </div>

            <div className="grid gap-4">
                {filteredBugs.length === 0 && (
                    <div className="flex flex-col items-center justify-center p-16 bg-slate-50 dark:bg-slate-900/50 border border-dashed border-slate-300 dark:border-slate-800 rounded-2xl">
                        <Bug size={48} className="text-slate-300 dark:text-slate-700 mb-4" />
                        <p className="text-slate-500 font-medium">No active bug reports found.</p>
                    </div>
                )}

                {filteredBugs.map(bug => (
                    <div key={bug.id} className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm transition-all hover:border-slate-300 dark:hover:border-slate-700">
                        <div className="flex justify-between items-start mb-4">
                            <div>
                                <div className="flex items-center gap-3 mb-1">
                                    <span className={`px-2 py-0.5 rounded text-[10px] font-bold uppercase border ${bug.severity === 'critical' ? 'bg-red-100 dark:bg-red-900/30 text-red-600 border-red-200 dark:border-red-900/50' :
                                        bug.severity === 'high' ? 'bg-orange-100 dark:bg-orange-900/30 text-orange-600 border-orange-200 dark:border-orange-900/50' :
                                            'bg-blue-100 dark:bg-blue-900/30 text-blue-600 border-blue-200 dark:border-blue-900/50'
                                        }`}>
                                        {bug.severity}
                                    </span>
                                    <h3 className="font-bold text-lg text-slate-900 dark:text-white">{bug.title}</h3>
                                </div>
                                <div className="flex items-center gap-2 text-xs text-slate-500">
                                    <span className="font-medium text-slate-700 dark:text-slate-300">{bug.userEmail}</span>
                                    <span>•</span>
                                    <span>{new Date(bug.timestamp).toLocaleString()}</span>
                                </div>
                            </div>

                            <div className="flex items-center gap-3">
                                <select
                                    value={bug.status}
                                    onChange={e => handleStatusChange(bug.id, e.target.value)}
                                    className={`bg-slate-100 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-xs font-bold rounded-lg px-3 py-2 text-slate-700 dark:text-slate-300 outline-none cursor-pointer hover:bg-slate-200 dark:hover:bg-slate-800 transition ${bug.status === 'resolved' ? 'text-green-600 dark:text-green-500' : ''
                                        }`}
                                >
                                    <option value="open">Open</option>
                                    <option value="in_progress">In Progress</option>
                                    <option value="resolved">Resolved</option>
                                </select>
                                <DoubleConfirmDeleteButton
                                    onDelete={() => handleDelete(bug.id)}
                                    className="p-2 text-slate-400 hover:text-red-500 hover:bg-red-50 dark:hover:bg-red-900/20 rounded-lg"
                                    title="Move to Trash"
                                />
                            </div>
                        </div>

                        <div className="bg-slate-50 dark:bg-slate-900 rounded-xl p-4 text-sm text-slate-700 dark:text-slate-300 font-medium border border-slate-100 dark:border-slate-800/50 whitespace-pre-wrap leading-relaxed">
                            {bug.description}
                        </div>

                        {bug.steps && (
                            <div className="mt-4 text-xs bg-slate-50/50 dark:bg-slate-950/30 p-3 rounded-lg border border-dashed border-slate-200 dark:border-slate-800">
                                <strong className="text-slate-500 uppercase tracking-wider block mb-1">Steps to Reproduce:</strong>
                                <p className="text-slate-600 dark:text-slate-400 whitespace-pre-wrap font-mono">{bug.steps}</p>
                            </div>
                        )}
                    </div>
                ))}
            </div>
        </div>
    );
};

const TutorialStepsManagement = () => {
    const [steps, setSteps] = useState<TutorialStep[]>([]);
    const [isEditing, setIsEditing] = useState<TutorialStep | null>(null);
    const [newStep, setNewStep] = useState<Partial<TutorialStep>>({
        title: '', description: '', actionLabel: '', actionLink: '', order: 0
    });
    const [isModalOpen, setIsModalOpen] = useState(false);

    useEffect(() => {
        loadSteps();
    }, []);

    const loadSteps = async () => {
        const data = await adminService.getTutorialSteps();
        setSteps(data.filter(s => !s.isDeleted));
    };

    const handleSave = async () => {
        if (!newStep.title) return;
        const stepToSave: TutorialStep = {
            id: isEditing ? isEditing.id : Math.random().toString(36).substr(2, 9),
            title: newStep.title,
            description: newStep.description || '',
            actionLabel: newStep.actionLabel,
            actionLink: newStep.actionLink,
            order: newStep.order || steps.length + 1,
            isDeleted: false
        };
        await adminService.saveTutorialStep(stepToSave);
        setIsModalOpen(false);
        setIsEditing(null);
        loadSteps();
    };

    const handleDelete = async (id: string) => {
        try {
            await adminService.deleteTutorialStep(id); // Soft delete
            await loadSteps();
        } catch (error) {
            console.error("Delete failed", error);
            alert("Failed to delete step");
        }
    };

    const openEdit = (step: TutorialStep) => {
        setIsEditing(step);
        setNewStep(step);
        setIsModalOpen(true);
    };

    const openNew = () => {
        setIsEditing(null);
        setNewStep({ title: '', description: '', actionLabel: '', actionLink: '', order: steps.length + 1 });
        setIsModalOpen(true);
    };

    const [draggedStepIndex, setDraggedStepIndex] = useState<number | null>(null);

    const handleDragStart = (e: React.DragEvent, index: number) => {
        setDraggedStepIndex(index);
        e.dataTransfer.effectAllowed = "move";
        // Make the drag image transparent or custom if needed, but default is usually fine
    };

    const handleDragOver = (e: React.DragEvent, index: number) => {
        e.preventDefault(); // Necessary to allow dropping
        e.dataTransfer.dropEffect = "move";

        if (draggedStepIndex === null || draggedStepIndex === index) return;

        // Reorder locally for visual feedback
        const newSteps = [...steps];
        const draggedStep = newSteps[draggedStepIndex];
        newSteps.splice(draggedStepIndex, 1);
        newSteps.splice(index, 0, draggedStep);

        setSteps(newSteps);
        setDraggedStepIndex(index);
    };

    const handleDragEnd = async () => {
        setDraggedStepIndex(null);
        // Persist the new order
        try {
            await adminService.reorderTutorialSteps(steps);
        } catch (error) {
            console.error("Failed to save new order", error);
            // Silently revert to previous order loaded from database
            loadSteps();
        }
    };

    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <div>
                    <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2">
                        <GraduationCap className="text-emerald-500" /> 5 Steps Tutorial Config
                    </h2>
                    <p className="text-sm text-slate-500 dark:text-slate-400">Manage the "Getting Started" widget in the user sidebar.</p>
                </div>
                <button onClick={openNew} className="px-4 py-2 bg-emerald-500 text-white font-bold rounded-lg hover:bg-emerald-600 transition flex items-center gap-2">
                    <Plus size={18} /> Add Step
                </button>
            </div>

            <div className="space-y-3">
                {steps.map((step, index) => (
                    <div
                        key={step.id}
                        draggable
                        onDragStart={(e) => handleDragStart(e, index)}
                        onDragOver={(e) => handleDragOver(e, index)}
                        onDragEnd={handleDragEnd}
                        className={`flex items-center gap-4 p-4 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl shadow-sm hover:border-emerald-500/50 transition group ${draggedStepIndex === index ? 'opacity-50 border-dashed border-emerald-500' : ''}`}
                    >
                        <div className="p-2 bg-slate-100 dark:bg-slate-800 rounded text-slate-400 cursor-grab active:cursor-grabbing">
                            <GripVertical size={16} />
                        </div>
                        <div className="w-8 h-8 rounded-full bg-emerald-100 dark:bg-emerald-500/10 text-emerald-600 dark:text-emerald-400 flex items-center justify-center font-bold text-sm border border-emerald-200 dark:border-emerald-500/20">
                            {index + 1}
                        </div>
                        <div className="flex-1">
                            <h3 className="font-bold text-slate-900 dark:text-white">{step.title}</h3>
                            <p className="text-sm text-slate-500 dark:text-slate-400">{step.description}</p>
                        </div>
                        {step.actionLabel && (
                            <div className="hidden md:block px-3 py-1 bg-slate-100 dark:bg-slate-800 rounded text-xs text-slate-600 dark:text-slate-300 font-mono">
                                [{step.actionLabel} {'->'} {step.actionLink}]
                            </div>
                        )}
                        <div className="flex gap-2">
                            <button onClick={() => openEdit(step)} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-blue-500 transition"><Edit size={16} /></button>
                            <DoubleConfirmDeleteButton onDelete={() => handleDelete(step.id)} className="p-2 hover:bg-red-100 dark:hover:bg-red-900/20 rounded-lg text-red-500 transition" />
                        </div>
                    </div>
                ))}
                {steps.length === 0 && (
                    <div className="text-center p-8 border-2 border-dashed border-slate-200 dark:border-slate-800 rounded-2xl text-slate-500">
                        No steps defined. Add one to get started.
                    </div>
                )}
            </div>

            {isModalOpen && (
                <div className="fixed inset-0 z-50 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4" onClick={() => setIsModalOpen(false)}>
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl w-full max-w-md p-6 shadow-2xl animate-fade-in-up" onClick={e => e.stopPropagation()}>
                        <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-4">{isEditing ? 'Edit Step' : 'Add Step'}</h3>
                        <div className="space-y-4">
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1">Step Title</label>
                                <input type="text" value={newStep.title} onChange={e => setNewStep({ ...newStep, title: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none focus:border-emerald-500" placeholder="e.g., Complete Profile" />
                            </div>
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1">Description</label>
                                <input type="text" value={newStep.description} onChange={e => setNewStep({ ...newStep, description: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none focus:border-emerald-500" placeholder="Short instruction..." />
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-1">Action Label (Opt)</label>
                                    <input type="text" value={newStep.actionLabel} onChange={e => setNewStep({ ...newStep, actionLabel: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none focus:border-emerald-500" placeholder="Go there" />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-1">Link ID (Opt)</label>
                                    <input type="text" value={newStep.actionLink} onChange={e => setNewStep({ ...newStep, actionLink: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none focus:border-emerald-500" placeholder="SETTINGS:api" />
                                </div>
                            </div>
                            <div className="flex justify-end gap-2 pt-4">
                                <button onClick={() => setIsModalOpen(false)} className="px-4 py-2 text-slate-500 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-sm font-bold">Cancel</button>
                                <button onClick={handleSave} className="px-4 py-2 bg-emerald-500 hover:bg-emerald-600 text-white rounded-lg text-sm font-bold">Save Step</button>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
};

const AuthConfiguration = () => {
    const [settings, setSettings] = useState<SiteSettings | null>(null);
    const [isSaving, setIsSaving] = useState(false);

    useEffect(() => {
        adminService.getSiteSettings().then(setSettings);
    }, []);

    const handleSave = async () => {
        if (settings) {
            setIsSaving(true);
            console.log('[BRANDING] Saving settings:', { appIcon: settings.appIcon, appLogo: settings.appLogo });
            await adminService.saveSiteSettings(settings);
            // Dispatch event to notify App.tsx to refresh
            window.dispatchEvent(new CustomEvent('sf_settings_updated'));
            setIsSaving(false);
            console.log('[BRANDING] Settings saved successfully');
        }
    };

    if (!settings) return <div className="p-10 flex justify-center"><Loader2 className="animate-spin" /></div>;
    const update = (key: keyof SiteSettings, val: any) => setSettings({ ...settings, [key]: val });
    const handlePathInput = (key: keyof SiteSettings, value: string) => {
        // Automatically fix local file paths if they point to the uploads directory
        // Example: "C:\Users\MyName\Desktop\Project\uploads\App\logo.png" -> "/uploads/App/logo.png"
        const match = value.match(/[\\/]uploads[\\/](.+)$/i);
        if (match) {
            // Found 'uploads/', convert to relative web path
            const relativePath = '/uploads/' + match[1].replace(/\\/g, '/');
            update(key, relativePath);
        } else {
            update(key, value);
        }
    };

    const ProviderConfig = ({ icon: Icon, colorClass, name, enabled, simulated, clientId, clientIdKey, simulatedKey, enabledKey, docsUrl }: any) => (
        <div className={`bg-white dark:bg-slate-900 border transition-all duration-300 rounded-2xl p-6 shadow-sm relative overflow-hidden ${enabled ? 'border-amber-500/30 ring-1 ring-amber-500/20' : 'border-slate-200 dark:border-slate-800'}`}>
            <div className="flex justify-between items-start mb-6 relative z-10">
                <div className="flex items-center gap-4">
                    <div className={`p-3 rounded-xl ${colorClass}`}><Icon size={24} /></div>
                    <div>
                        <h3 className="font-bold text-slate-900 dark:text-white text-lg flex items-center gap-2">{name} Login {enabled && (<span className={`text-[10px] px-2 py-0.5 rounded-full border font-bold uppercase tracking-wider ${simulated ? 'bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 border-blue-200 dark:border-blue-500/30' : 'bg-green-100 dark:bg-green-900/30 text-green-600 dark:text-green-400 border-green-200 dark:border-green-500/30'}`}>{simulated ? 'Dev Mode' : 'Live'}</span>)}</h3>
                        <p className="text-xs text-slate-500 dark:text-slate-400 mt-1">{enabled ? (simulated ? 'Using internal simulator' : 'Using production OAuth provider') : 'Disabled for users'}</p>
                    </div>
                </div>
                <Toggle checked={enabled} onChange={() => update(enabledKey, !enabled)} />
            </div>
            {enabled && (<div className="animate-fade-in bg-slate-50 dark:bg-slate-950/50 rounded-xl p-5 border border-slate-100 dark:border-slate-800/50"><div className="flex gap-2 mb-6 p-1 bg-slate-200 dark:bg-slate-800 rounded-lg w-fit"><button onClick={() => update(simulatedKey, true)} className={`px-4 py-2 rounded-md text-xs font-bold flex items-center gap-2 transition-all ${simulated ? 'bg-white dark:bg-slate-700 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}><Terminal size={14} /> Simulation (Dev)</button><button onClick={() => update(simulatedKey, false)} className={`px-4 py-2 rounded-md text-xs font-bold flex items-center gap-2 transition-all ${!simulated ? 'bg-white dark:bg-slate-700 text-slate-900 dark:text-white shadow-sm' : 'text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}><Globe size={14} /> Production (Live)</button></div>{!simulated ? (<div className="space-y-4 animate-fade-in"><div><div className="flex justify-between items-center mb-2"><label className="text-xs font-bold text-slate-500 uppercase flex items-center gap-1">Client ID / App ID <span className="text-red-500">*</span></label><a href={docsUrl} target="_blank" rel="noreferrer" className="text-[10px] text-blue-500 hover:underline flex items-center gap-1">Get ID <ExternalLink size={10} /></a></div><div className="relative"><Key className="absolute left-3 top-3 text-slate-400" size={16} /><input type="text" value={clientId || ''} onChange={e => update(clientIdKey, e.target.value)} className="w-full bg-white dark:bg-slate-900 border border-slate-300 dark:border-slate-700 rounded-xl py-2.5 pl-10 pr-4 text-sm font-mono outline-none focus:border-amber-500 transition-colors" placeholder={`Enter ${name} Client ID`} /></div>{!clientId && <p className="text-[10px] text-red-500 mt-1 flex items-center gap-1"><AlertCircle size={10} /> Required for live login</p>}</div><div className="p-3 bg-blue-50 dark:bg-blue-900/10 border border-blue-100 dark:border-blue-900/30 rounded-lg flex gap-3 items-start"><Info className="text-blue-500 shrink-0 mt-0.5" size={16} /><p className="text-xs text-blue-700 dark:text-blue-300 leading-relaxed">Ensure your authorized redirect URIs in the {name} Console match your domain: <code className="bg-blue-100 dark:bg-blue-900/40 px-1 rounded mx-1">https://your-domain.com/auth/callback</code></p></div></div>) : (<div className="text-xs text-slate-500 italic p-2 flex items-center gap-2"><CheckCircle2 size={14} className="text-green-500" /> Using internal mock authentication flow. No keys required.</div>)}</div>)}
        </div>
    );

    return (
        <div className="space-y-8 max-w-6xl">
            {/* App Branding Section */}
            <div className="space-y-4 mb-8">
                <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                    <div>
                        <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2">
                            <Palette className="text-amber-500" /> App Branding
                        </h2>
                        <p className="text-slate-500 dark:text-slate-400 text-sm mt-1">Customize app icon (favicon) and logo displayed throughout the application.</p>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {/* App Icon (Favicon) */}
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm">
                        <div className="flex items-center gap-3 mb-4">
                            <div className="p-3 rounded-xl bg-blue-100 dark:bg-blue-900/20 text-blue-600 dark:text-blue-400">
                                <Camera size={24} />
                            </div>
                            <div>
                                <h3 className="font-bold text-slate-900 dark:text-white text-lg">App Icon (Favicon)</h3>
                                <p className="text-xs text-slate-500 dark:text-slate-400">Displays in browser tab (16x16 or 32x32 recommended)</p>
                            </div>
                        </div>

                        <div className="space-y-3">
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Icon URL</label>
                                <input
                                    type="text"
                                    value={settings.appIcon || ''}
                                    onChange={e => handlePathInput('appIcon', e.target.value)}
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500"
                                    placeholder="https://example.com/favicon.ico or /uploads/favicon.ico"
                                />
                            </div>
                            {settings.appIcon && (
                                <div className="flex items-center gap-2 p-3 bg-slate-50 dark:bg-slate-950 rounded-lg border border-slate-200 dark:border-slate-800">
                                    <img src={settings.appIcon} alt="App Icon" className="w-8 h-8" style={{ imageRendering: 'pixelated' }} />
                                    <span className="text-xs text-slate-500">Preview</span>
                                </div>
                            )}
                            <p className="text-[10px] text-slate-500">💡 Tip: Upload to /uploads folder or use external URL</p>
                        </div>
                    </div>

                    {/* App Logo */}
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm">
                        <div className="flex items-center gap-3 mb-4">
                            <div className="p-3 rounded-xl bg-purple-100 dark:bg-purple-900/20 text-purple-600 dark:text-purple-400">
                                <ImageIcon size={24} />
                            </div>
                            <div>
                                <h3 className="font-bold text-slate-900 dark:text-white text-lg">App Logo</h3>
                                <p className="text-xs text-slate-500 dark:text-slate-400">Displays in header/sidebar (any size, transparent PNG recommended)</p>
                            </div>
                        </div>

                        <div className="space-y-3">
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Logo URL</label>
                                <input
                                    type="text"
                                    value={settings.appLogo || ''}
                                    onChange={e => handlePathInput('appLogo', e.target.value)}
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm outline-none focus:border-amber-500"
                                    placeholder="https://example.com/logo.png or /uploads/logo.png"
                                />
                            </div>
                            {settings.appLogo && (
                                <div className="flex items-center gap-2 p-3 bg-slate-50 dark:bg-slate-950 rounded-lg border border-slate-200 dark:border-slate-800">
                                    <img src={settings.appLogo} alt="App Logo" className="h-8 object-contain" />
                                    <span className="text-xs text-slate-500">Preview</span>
                                </div>
                            )}
                            <p className="text-[10px] text-slate-500">💡 Tip: Use transparent PNG for best results</p>
                        </div>
                    </div>
                </div>
            </div>

            {/* Authentication Providers Section */}
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4"><div><h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><ShieldCheck className="text-amber-500" /> Authentication Providers</h2><p className="text-slate-500 dark:text-slate-400 text-sm mt-1">Configure social login strategies and toggle between development simulation and live production modes.</p></div><button onClick={handleSave} disabled={isSaving} className="px-6 py-2 bg-amber-500 hover:bg-amber-600 text-slate-950 font-bold rounded-xl transition flex items-center gap-2 disabled:opacity-50 shadow-lg shadow-amber-500/20">{isSaving ? <Loader2 className="animate-spin" size={18} /> : <Save size={18} />} Save Configuration</button></div>
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6"><ProviderConfig name="Google" icon={Globe} colorClass="bg-red-100 dark:bg-red-900/20 text-red-600 dark:text-red-400" enabled={settings.enableGoogleLogin} simulated={settings.enableSimulatedGoogleLogin} clientId={settings.googleClientId} clientIdKey="googleClientId" simulatedKey="enableSimulatedGoogleLogin" enabledKey="enableGoogleLogin" docsUrl="https://console.cloud.google.com/apis/credentials" /><ProviderConfig name="Facebook" icon={Facebook} colorClass="bg-blue-100 dark:bg-blue-900/20 text-blue-600 dark:text-blue-400" enabled={settings.enableFacebookLogin} simulated={settings.enableSimulatedFacebookLogin} clientId={settings.facebookClientId} clientIdKey="facebookClientId" simulatedKey="enableSimulatedFacebookLogin" enabledKey="enableFacebookLogin" docsUrl="https://developers.facebook.com/apps/" /><ProviderConfig name="Twitter" icon={Twitter} colorClass="bg-sky-100 dark:bg-sky-900/20 text-sky-600 dark:text-sky-400" enabled={settings.enableTwitterLogin} simulated={settings.enableSimulatedTwitterLogin} clientId={settings.twitterClientId} clientIdKey="twitterClientId" simulatedKey="enableSimulatedTwitterLogin" enabledKey="enableTwitterLogin" docsUrl="https://developer.twitter.com/en/portal/dashboard" /><div className="border border-dashed border-slate-300 dark:border-slate-700 rounded-2xl p-6 flex flex-col items-center justify-center text-center text-slate-400 hover:bg-slate-50 dark:hover:bg-slate-900/50 transition group cursor-pointer"><div className="w-12 h-12 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center mb-3 group-hover:scale-110 transition-transform"><Plus size={24} /></div><h3 className="font-bold text-sm">Add Provider</h3><p className="text-xs mt-1">GitHub, LinkedIn, SSO...</p></div></div>
        </div>
    );
};

const TipsManagement = () => {
    const [tips, setTips] = useState<Tip[]>([]);
    const [isEditing, setIsEditing] = useState<Tip | null>(null);
    const [newTip, setNewTip] = useState<Partial<Tip>>({ title: '', text: '', category: '' });
    const [isModalOpen, setIsModalOpen] = useState(false);

    useEffect(() => { loadTips(); }, []);
    const loadTips = async () => {
        const data = await adminService.getAllTips();
        setTips(data.filter(t => !t.isDeleted)); // Only show active
    };
    const handleSave = async () => {
        if (!newTip.title || !newTip.text) return;
        const tipToSave: Tip = {
            id: isEditing ? isEditing.id : Math.random().toString(36).substr(2, 9),
            title: newTip.title,
            text: newTip.text,
            category: newTip.category || 'General',
            createdAt: isEditing ? isEditing.createdAt : Date.now(),
            isDeleted: false
        };
        await adminService.saveTip(tipToSave); setIsModalOpen(false); setNewTip({ title: '', text: '', category: '' }); setIsEditing(null); loadTips();
    };
    const handleDelete = async (id: string) => {
        try {
            await adminService.deleteTip(id);
            await loadTips();
        } catch (error) {
            console.error("Delete failed", error);
            alert("Failed to delete tip");
        }
    };
    const openEdit = (tip: Tip) => { setIsEditing(tip); setNewTip({ title: tip.title, text: tip.text, category: tip.category }); setIsModalOpen(true); };
    const openNew = () => { setIsEditing(null); setNewTip({ title: '', text: '', category: '' }); setIsModalOpen(true); }

    const handlePopulate = async () => {
        if (confirm("This will add default tips to your database. Continue?")) {
            await adminService.populateDefaultTips();
            await loadTips();
        }
    };

    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><Lightbulb className="text-amber-500" /> Tips of the Day Management</h2>
                <div className="flex gap-2">
                    <button onClick={handlePopulate} className="px-4 py-2 bg-slate-100 dark:bg-slate-800 text-slate-600 dark:text-slate-300 font-bold rounded-lg hover:bg-slate-200 dark:hover:bg-slate-700 transition flex items-center gap-2"><Database size={18} /> Populate Defaults</button>
                    <button onClick={openNew} className="px-4 py-2 bg-amber-500 text-slate-950 font-bold rounded-lg hover:bg-amber-600 transition flex items-center gap-2"><Plus size={18} /> Add New Tip</button>
                </div>
            </div>
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden shadow-sm dark:shadow-none"><table className="w-full text-left text-sm text-slate-600 dark:text-slate-400"><thead className="bg-slate-50 dark:bg-slate-950 text-slate-500 uppercase text-xs border-b border-slate-200 dark:border-slate-800"><tr><th className="p-4">Title</th><th className="p-4">Content</th><th className="p-4">Category</th><th className="p-4">Date Added</th><th className="p-4 text-right">Actions</th></tr></thead><tbody>{tips.map(tip => (<tr key={tip.id} className="border-t border-slate-200 dark:border-slate-800 hover:bg-slate-50 dark:hover:bg-slate-800/50"><td className="p-4 font-bold text-slate-900 dark:text-white">{tip.title}</td><td className="p-4 truncate max-w-xs" title={tip.text}>{tip.text}</td><td className="p-4"><span className="px-2 py-1 bg-slate-100 dark:bg-slate-800 rounded text-xs font-bold">{tip.category}</span></td><td className="p-4">{new Date(tip.createdAt).toLocaleDateString()}</td><td className="p-4 flex justify-end gap-2"><button onClick={() => openEdit(tip)} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-blue-500 transition"><Edit size={16} /></button><DoubleConfirmDeleteButton onDelete={() => handleDelete(tip.id)} className="p-2 hover:bg-red-100 dark:hover:bg-red-900/20 rounded-lg text-red-500 transition" /></td></tr>))}{tips.length === 0 && (<tr><td colSpan={5} className="p-8 text-center">No tips found. Add some to engage users!</td></tr>)}</tbody></table></div>
            {isModalOpen && (<div className="fixed inset-0 z-50 bg-black/60 backdrop-blur-sm flex items-center justify-center p-4" onClick={() => setIsModalOpen(false)}><div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl w-full max-w-md p-6 shadow-2xl animate-fade-in-up" onClick={e => e.stopPropagation()}><h3 className="text-xl font-bold text-slate-900 dark:text-white mb-4">{isEditing ? 'Edit Tip' : 'Add New Tip'}</h3><div className="space-y-4"><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Title</label><input type="text" value={newTip.title} onChange={e => setNewTip({ ...newTip, title: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none focus:border-amber-500" placeholder="Tip Headline" /></div><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Category</label><input type="text" value={newTip.category} onChange={e => setNewTip({ ...newTip, category: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none focus:border-amber-500" placeholder="e.g. Video, Strategy, Image" /></div><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1">Content</label><textarea value={newTip.text} onChange={e => setNewTip({ ...newTip, text: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-lg p-2 text-sm text-slate-900 dark:text-white outline-none focus:border-amber-500 h-32 resize-none" placeholder="Enter the tip details..." /></div><div className="flex justify-end gap-2 pt-2"><button onClick={() => setIsModalOpen(false)} className="px-4 py-2 text-slate-500 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg text-sm font-bold">Cancel</button><button onClick={handleSave} className="px-4 py-2 bg-amber-500 hover:bg-amber-600 text-slate-950 rounded-lg text-sm font-bold">Save Tip</button></div></div></div></div>)}
        </div>
    );
};

const SiteConfiguration = () => {
    const [settings, setSettings] = useState<SiteSettings | null>(null);
    const [isSaving, setIsSaving] = useState(false);
    const [timeToReset, setTimeToReset] = useState('');

    // Default prompts from geminiService.ts
    const DEFAULT_ANATOMY_PROMPT = `🚨🚨🚨 ULTRA-CRITICAL: HUMAN ANATOMY DEFECT DETECTION 🚨🚨🚨
    ZERO-TOLERANCE POLICY. ANY DEFECT = INSTANT REJECTION.

⚠️ WARNING: THIS IS AN ADVERSARIAL TEST. THE IMAGE LIKELY HAS 6 FINGERS. ⚠️
ASSUME EVERY HAND HAS 6 FINGERS UNTIL YOU COUNT EXACTLY 5.

═══════════════════════════════════════════════════════

STEP 1: Detect if humans are present-If NO humans → score: 100, STOP HERE-If humans present → MANDATORY VERIFICATION REQUIRED

═══════════════════════════════════════════════════════

STEP 2: 🔴 MANDATORY FINGER ENUMERATION PROTOCOL 🔴

FOR EACH VISIBLE HAND, YOU MUST:
A) PAUSE and look ONLY at that hand. ZOOM IN MENTALLY.
B) Starting from the THUMB, count and NAME each digit:

   Example for a CORRECT hand:
   1. Thumb (short, thick)
   2. Index finger
   3. Middle finger (longest)
   4. Ring finger
   5. Pinky finger (smallest)
   TOTAL = 5 digits ✓ PASS

   Example of WRONG hand (6 fingers):
   1. Thumb
   2. Index finger
   3. Middle finger
   4. Ring finger
   5. Extra finger (Look closely between ring/pinky!)
   6. Pinky finger
   TOTAL = 6 digits ✗ REJECT IMMEDIATELY

C) RULES FOR REJECTION:
   ✗ 6+ digits on any hand → score: 0 (REJECT)
   ✗ 4 or fewer digits (when hand is fully visible) → score: 0 (REJECT)
   ✗ Fingers appear fused/merged → score: 0 (REJECT)
   ✗ Extra thumb or double thumb → score: 0 (REJECT)
   ✗ Finger in impossible position → score: 0 (REJECT)

D) FINAL CHECK:
   - "Am I 100% sure there are only 5 fingers?"
   - "Is there a hidden 6th finger?"
   - IF ANY DOUBT → SCORE: 0

⚠️ CRITICAL: If you see MORE than 5 digits on a hand, you MUST return score: 0
⚠️ DO NOT MAKE EXCUSES-6 fingers = AUTOMATIC REJECTION

═══════════════════════════════════════════════════════

STEP 3: 🔴 CHECK FACE FOR ARTIFACTS (if fingers passed) 🔴
- Distorted/warped features → score: 0-AI artifacts/glitches → score: 0-Asymmetry (severely lopsided) → score: 0-Duplicated features → score: 0-Missing features → score: 0-Unnatural merging → score: 0-Texture/color errors → score: 0

═══════════════════════════════════════════════════════

STEP 4: 🔴 CHECK EYES FOR DEFECTS (if all passed) 🔴
- Pupil size mismatch → score: 0-Wrong eye count (not 2) → score: 0-Blurred/corrupted eyes → score: 0-Drastically different shapes → score: 0-Missing pupils/iris → score: 0-Duplicated/overlapping eyes → score: 0

═══════════════════════════════════════════════════════

STEP 5: 🔴 COUNT LIMBS (if all passed) 🔴
- Each person must have EXACTLY 2 arms-Each person must have EXACTLY 2 legs-3+ arms on one person → score: 0-3+ legs on one person → score: 0-Floating extra limbs → score: 0

═══════════════════════════════════════════════════════

STEP 6: 🔴 COUNT TOES (ONLY if barefoot) 🔴
- If wearing shoes/socks → SKIP-If barefoot → each foot must have EXACTLY 5 toes-6 toes or 4 toes → score: 0

═══════════════════════════════════════════════════════

FINAL SCORING:
- ANY defect found = score: 0 (REJECT)
- NO defects found = score: 100 (PASS)
- When uncertain = score: 0 (REJECT)

🚨 REPEAT REMINDER 🚨
- 6 FINGERS = SCORE 0 (NO EXCEPTIONS!)
- You MUST count fingers explicitly-You MUST list the count in your analysis-DO NOT give score: 100 without counting fingers first

OUTPUT MUST BE VALID JSON (no extra text):
{
  "score": 0 or 100,
  "hasHumans": true or false,
  "defects": ["list defects found"],
  "analysis": "MUST include: 'Counted X fingers on visible hand(s)' if humans present"
}

🔴 CRITICAL: If you counted 6 fingers, defects MUST include "Extra finger detected-6 fingers on hand" and score MUST be 0`;

    const DEFAULT_QUALITY_PROMPT = `CRITICAL TASK: TECHNICAL QUALITY ASSESSMENT for stock photography standards(Adobe Stock, Shutterstock, iStock).
This is a STRICT QUALITY CONTROL CHECK.Be CONSISTENT and OBJECTIVE in your scoring.

    IMPORTANT: Analyze this image for TECHNICAL QUALITY ONLY.Do NOT consider anatomy or content-only technical aspects.

EVALUATION CRITERIA(Score each 0-10, then multiply):

1. NOISE / GRAIN(×10 points):
- 10: Perfectly clean, no visible noise even at 100 % zoom-8-9: Very slight noise only visible at high magnification-6-7: Moderate noise visible at normal viewing-4-5: Significant noise affecting image quality-2-3: Heavy noise, grainy appearance-0-1: Extreme noise, unusable quality

2. SHARPNESS / FOCUS(×10 points):
- 10: Perfectly sharp, crisp details throughout-8-9: Very sharp, minor softness in non-critical areas-6-7: Acceptable sharpness, some soft areas-4-5: Noticeably soft or slight motion blur-2-3: Significantly blurred or out of focus-0-1: Completely blurred, unusable

3. EXPOSURE(×10 points):
- 10: Perfect exposure, excellent dynamic range-8-9: Well exposed with minor highlight / shadow clipping-6-7: Acceptable exposure, some detail loss-4-5: Overexposed or underexposed areas-2-3: Significant exposure problems-0-1: Completely blown out or too dark

4. COLOR ACCURACY(×10 points):
- 10: Natural, accurate colors, no color casts-8-9: Very good color with minor variations-6-7: Acceptable color, slight color cast-4-5: Noticeable color issues or unnatural saturation-2-3: Strong color casts or poor color balance-0-1: Completely wrong colors

5. COMPOSITION(×10 points):
- 10: Professional composition, rule of thirds, perfect balance-8-9: Very good composition, well-balanced-6-7: Acceptable composition, minor issues-4-5: Poor composition, unbalanced-2-3: Very poor composition-0-1: Chaotic, no composition structure

6. LIGHTING QUALITY(×10 points):
- 10: Professional lighting, perfect highlights and shadows-8-9: Very good lighting, well-balanced-6-7: Acceptable lighting, minor issues-4-5: Harsh shadows or poor lighting-2-3: Very poor lighting quality-0-1: Completely inadequate lighting

7. TECHNICAL DEFECTS(×10 points):
- 10: No artifacts, compression, or defects-8-9: Very minor compression artifacts-6-7: Visible but acceptable compression-4-5: Noticeable compression or artifacts-2-3: Significant technical defects-0-1: Severe artifacts or corruption

SCORING CALCULATION:
1. Evaluate each criterion and assign a score(0-10)
2. Average all 7 scores
3. Multiply by 10 to get final score(0-100)
4. Round to nearest whole number

CONSISTENCY RULES:
- Be objective and consistent-Same image should always get same score-Focus on measurable technical qualities-Ignore content, only judge technical quality

OUTPUT JSON ONLY:
{
    "score": 0-100,
        "issues": ["list of specific technical issues found"],
            "reasoning": "detailed breakdown of each criterion score"
} `;

    // AI Reviewer Prompts State
    const DEFAULT_ADOBE_INSTRUCTION = `Determine the most appropriate "category" for this image.
    CRITICAL: The category MUST be exactly one of the values from this list (choose the best match):
    Animals, Buildings and Architecture, Business, Drinks, The Environment, States of Mind, Food, Graphic Resources, Hobbies and Leisure, Industry, Landscapes, Lifestyle, People, Plants and Flowers, Culture and Religion, Science, Social Issues, Sports, Technology, Transport, Travel.`;

    const DEFAULT_TITLE_INSTRUCTION = "Engaging and descriptive.";
    const DEFAULT_DESC_INSTRUCTION = "Write a detailed, rich description that fills this space.";
    const DEFAULT_KEYWORD_INSTRUCTION = "Provide a mix of common and long-tail keywords relevant to the subject.";

    const DEFAULT_KEYWORD_GENERATION_PROMPT = "🚨 CRITICAL PRIORITY 🚨 - You MUST generate EXACTLY " + "{{" + "keywordCount" + "}}" + " keywords TOTAL when combining 'commonKeywords' and 'longTailKeywords'.\n\n📊 COUNT ENFORCEMENT (NON-NEGOTIABLE):\n- ⚠️ MANDATORY: commonKeywords.length + longTailKeywords.length MUST BE BETWEEN " + "{{" + "keywordCount - 3" + "}}" + " AND " + "{{" + "keywordCount + 10" + "}}" + "\n- EXAMPLE: User wants 50 keywords → You MUST generate 47-60 keywords. Generating 20-25 is COMPLETELY UNACCEPTABLE\n- IT IS BETTER TO HAVE " + "{{" + "keywordCount + 10" + "}}" + " KEYWORDS THAN " + "{{" + "keywordCount - 10" + "}}" + " KEYWORDS. AIM HIGH, NOT LOW.\n\n✅ KEYWORD SEO ORDER (CRITICAL FOR STOCK PLATFORMS):\nFirst 5 keywords are MOST IMPORTANT for ranking. Order keywords by priority:\n1. Main subject (what is it?)\n2. Secondary subject (what else is in the image?)\n3. Usage context (background, copy space, banner, header)\n4. Concept/emotion (happiness, freedom, success, celebration)\n5. Descriptive attributes (colors, lighting, composition)\n6. Style (modern, vintage, minimalist, realistic)\n7. Location/season (urban, beach, summer, winter, indoor, outdoor)\n\n🔟 NEGATIVE CONTROL (What to AVOID):\n❌ Wrong/unrelated locations\n❌ Objects not present in the image\n❌ Famous brand names or trademarks\n❌ Spam repetition or duplicate keywords\n❌ Misleading or irrelevant terms that hurt SEO ranking\n\n💡 KEYWORD GENERATION STRATEGY:\n- Think like a BUYER searching for this image, not the creator\n- Use highly relevant, specific keywords\n- NO duplicates or near-duplicates\n- When low on ideas: Add color variations (blue, azure, navy, cerulean), lighting (bright, golden hour, dusk, soft light), mood (cheerful, serene, energetic, peaceful), textures (smooth, rough, grainy), composition (centered, symmetrical, rule-of-thirds), style (modern, vintage, minimalist), seasonal (summer, winter, festive), contextual (indoor, outdoor, urban, rural), conceptual (freedom, growth, innovation, teamwork)\n\n🎯 FINAL VERIFICATION BEFORE RESPONDING:\nCount commonKeywords.length + longTailKeywords.length. If not between " + "{{" + "keywordCount - 3" + "}}" + " and " + "{{" + "keywordCount + 10" + "}}" + ", ADD MORE KEYWORDS IMMEDIATELY using synonyms and related concepts.";

    const [configTab, setConfigTab] = useState<'general' | 'microstock' | 'reviewer'>('general');
    const [anatomyPrompt, setAnatomyPrompt] = useState(localStorage.getItem('sf_reviewer_prompt_anatomy_v3') || DEFAULT_ANATOMY_PROMPT);
    const [qualityPrompt, setQualityPrompt] = useState(localStorage.getItem('sf_reviewer_prompt_quality_v3') || DEFAULT_QUALITY_PROMPT);

    // AI Reviewer Threshold Settings
    const [anatomyPassScore, setAnatomyPassScore] = useState(Number(localStorage.getItem('sf_reviewer_anatomy_pass') || 100));
    const [anatomyFailScore, setAnatomyFailScore] = useState(Number(localStorage.getItem('sf_reviewer_anatomy_fail') || 0));
    const [qualityMinScore, setQualityMinScore] = useState(Number(localStorage.getItem('sf_reviewer_quality_min') || 60));
    const [totalScoreThreshold, setTotalScoreThreshold] = useState(Number(localStorage.getItem('sf_reviewer_total_threshold') || 160));

    useEffect(() => {
        adminService.getSiteSettings().then(s => {
            // Initialize sidebar settings if missing or incomplete
            const defaultSidebar: { [key: string]: any } = s.sidebarSettings || {};

            const allRoutes = [
                'DASHBOARD', 'CHAT_INTELLIGENCE', 'MEDIA_ANALYSIS',
                'IMAGE_GENERATION', 'IMAGE_EDITOR', 'VIDEO_GENERATION', 'REVIEWER_AI',
                'LIBRARY', 'FAVORITES', 'TRASH'
            ];

            let hasUpdates = false;
            allRoutes.forEach(id => {
                if (!defaultSidebar[id]) {
                    defaultSidebar[id] = { id, enabled: true, comingSoon: false };
                    hasUpdates = true;
                }
            });

            if (hasUpdates || !s.sidebarSettings) {
                setSettings({ ...s, sidebarSettings: defaultSidebar });
            } else {
                setSettings(s);
            }
        });

        const updateTimer = () => {
            const now = new Date();

            // Calculate next midnight in Pacific Time
            // We create a date object representing "Now" in PT
            const ptNowString = now.toLocaleString('en-US', { timeZone: 'America/Los_Angeles' });
            const ptNow = new Date(ptNowString);

            // Create PT Midnight object
            const ptMidnight = new Date(ptNow);
            ptMidnight.setDate(ptMidnight.getDate() + 1);
            ptMidnight.setHours(0, 0, 0, 0);

            const diff = ptMidnight.getTime() - ptNow.getTime();

            if (diff <= 0) {
                setTimeToReset("00:00:00");
                return;
            }

            const h = Math.floor(diff / (1000 * 60 * 60));
            const m = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
            const s = Math.floor((diff % (1000 * 60)) / 1000);

            setTimeToReset(`${h.toString().padStart(2, '0')}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`);
        };

        updateTimer();
        const interval = setInterval(updateTimer, 1000);
        return () => clearInterval(interval);
    }, []);

    const handleToggle = (key: keyof SiteSettings) => {
        if (settings) setSettings({ ...settings, [key]: !settings[key as any] });
    };

    const handleModelLimitChange = (modelKey: string, value: number) => {
        if (settings && settings.modelLimits) {
            setSettings({
                ...settings,
                modelLimits: {
                    ...settings.modelLimits,
                    [modelKey]: value
                }
            });
        }
    };

    const handleSidebarToggle = (key: string, field: 'enabled' | 'comingSoon') => {
        if (settings && settings.sidebarSettings) {
            const updatedSidebarSettings = {
                ...settings.sidebarSettings,
                [key]: {
                    ...settings.sidebarSettings[key],
                    [field]: !settings.sidebarSettings[key][field]
                }
            };
            setSettings({ ...settings, sidebarSettings: updatedSidebarSettings });
        }
    };

    const handleSave = async () => {
        if (settings) {
            setIsSaving(true);
            const total = MODELS.reduce((sum, model) => {
                const key = MODEL_TYPE_TO_SETTINGS[model.id];
                return sum + (settings.modelLimits[key] || 0);
            }, 0);
            const updatedSettings = { ...settings, creditsPerApiKey: total };
            await adminService.saveSiteSettings(updatedSettings);
            setSettings(updatedSettings);
            setIsSaving(false);
        }
    };

    // Calculate total credits based ONLY on the visible/enabled models
    // DEFAULT TIER LIMITS (Static Hardcoded)
    const DEFAULT_TIER_LIMITS: Record<string, { free: number, paid: number }> = {
        'pro_3_0': { free: 0, paid: 1000 },
        'pro_2_5': { free: 0, paid: 1000 },
        'flash_2_5': { free: 20, paid: 1000 },
        'flash_lite_2_5': { free: 20, paid: 1000 },
        'imagen_ultra': { free: 10, paid: 500 },
        'imagen_fast': { free: 10, paid: 500 },
        'imagen_std': { free: 10, paid: 500 },
        'gemini_3_pro_image': { free: 50, paid: 500 },
        'gemini_2_5_flash_image': { free: 100, paid: 500 },
        'veo_3_1': { free: 20, paid: 500 },
        'veo_3_1_fast': { free: 50, paid: 500 },
        'veo_3_0': { free: 10, paid: 500 },
        'veo_2_0': { free: 5, paid: 500 },
        'gemini_3_0_flash': { free: 20, paid: 1000 },
        'gemini_1_5_pro': { free: 0, paid: 1000 },
    };

    // Calculate total credits based ONLY on the visible/enabled models
    const totals = useMemo(() => {
        if (!settings) return { free: 0, paid: 0 };
        return MODELS.reduce((acc, model) => {
            const key = MODEL_TYPE_TO_SETTINGS[model.id];

            // Helper to determine credit value
            const getModelLimit = (type: 'free' | 'paid') => {
                // Check if toggle is explicitly disabled
                if (settings.modelAccessTiers?.[model.id]?.[type] === false) return 0;

                // Get value -> Setting -> Default -> 0
                return settings.modelLimitsByTier?.[key]?.[type] ?? DEFAULT_TIER_LIMITS[key]?.[type] ?? 0;
            };

            return {
                free: acc.free + getModelLimit('free'),
                paid: acc.paid + getModelLimit('paid')
            };
        }, { free: 0, paid: 0 });
    }, [settings]);

    if (!settings) return <div>Loading...</div>;

    return (
        <div className="space-y-8">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><Sliders className="text-amber-500" /> General App Configuration</h2>
                <button onClick={handleSave} disabled={isSaving} className="px-6 py-2 bg-amber-500 text-slate-950 font-bold rounded-lg hover:bg-amber-600 transition disabled:opacity-50">{isSaving ? 'Saving...' : 'Save Changes'}</button>
            </div>
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-8 space-y-6 max-w-6xl shadow-sm dark:shadow-none">
                {/* Tabs */}
                <div className="flex gap-6 border-b border-slate-200 dark:border-slate-800 mb-2">
                    <button
                        onClick={() => setConfigTab('general')}
                        className={`pb-3 text-sm font-bold border-b-2 transition ${configTab === 'general' ? 'border-amber-500 text-amber-600 dark:text-amber-500' : 'border-transparent text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                    >
                        General Settings
                    </button>
                    <button
                        onClick={() => setConfigTab('microstock')}
                        className={`pb-3 text-sm font-bold border-b-2 transition ${configTab === 'microstock' ? 'border-amber-500 text-amber-600 dark:text-amber-500' : 'border-transparent text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                    >
                        Microstock Setup
                    </button>
                    <button
                        onClick={() => setConfigTab('reviewer')}
                        className={`pb-3 text-sm font-bold border-b-2 transition ${configTab === 'reviewer' ? 'border-amber-500 text-amber-600 dark:text-amber-500' : 'border-transparent text-slate-500 hover:text-slate-700 dark:hover:text-slate-300'}`}
                    >
                        AI Reviewer
                    </button>
                </div>

                {/* General Settings: Maintenance, Registrations, Email */}
                <div style={{ display: configTab === 'general' ? 'block' : 'none' }}>
                    <div className="flex items-center justify-between pb-6 border-b border-slate-200 dark:border-slate-800 mb-6"><div><h3 className="font-bold text-slate-900 dark:text-white">Maintenance Mode</h3><p className="text-sm text-slate-500 dark:text-slate-400">Disable user access for updates.</p></div><Toggle checked={settings.maintenanceMode} onChange={() => handleToggle('maintenanceMode')} /></div>
                    <div className="flex items-center justify-between pb-6 border-b border-slate-200 dark:border-slate-800 mb-6"><div><h3 className="font-bold text-slate-900 dark:text-white">Allow Registrations</h3><p className="text-sm text-slate-500 dark:text-slate-400">New users can sign up.</p></div><Toggle checked={settings.allowRegistrations} onChange={() => handleToggle('allowRegistrations')} /></div>
                    <div className="flex items-center justify-between pb-6 border-b border-slate-200 dark:border-slate-800"><div><h3 className="font-bold text-slate-900 dark:text-white">Email Verification</h3><p className="text-sm text-slate-500 dark:text-slate-400">Require email check before login.</p></div><Toggle checked={settings.emailVerificationRequired} onChange={() => handleToggle('emailVerificationRequired')} /></div>
                </div>

                {/* Microstock Settings: Adobe Stock */}
                {configTab === 'microstock' && (
                    <div className="space-y-6 pt-4">
                        <div>
                            <h3 className="font-bold text-slate-900 dark:text-white">General Metadata Instructions</h3>
                            <p className="text-sm text-slate-500 dark:text-slate-400 mb-4">Set specific instructions for how the AI generates each metadata field.</p>

                            <div className="grid grid-cols-1 gap-4">
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Image Title Instructions</label>
                                    <textarea
                                        value={settings.titleInstructionImage || (settings.titleInstruction || `Act as a top-ranked Adobe Stock contributor and SEO specialist.
Visually analyze the media and extract:
1. Primary subject (must be first in title)
2. Supporting visual elements
3. Environment or background
4. Commercial usage intent
5. Conceptual meaning (if relevant)

Create ONE SEO title that:
- matches buyer search intent
- is descriptive, not poetic
- starts with the main subject
- uses clear, searchable keywords
- includes "background", "copy space", or "template" ONLY if visible
- avoids adjectives that do not describe visuals
- avoids emotional storytelling
- avoids branding and metaphors
- is concise, professional, and under 70 characters`)}
                                        onChange={(e) => setSettings({ ...settings, titleInstructionImage: e.target.value })}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-64 resize-none font-mono"
                                        placeholder="Instruction for Image Title generation..."
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Video Title Instructions</label>
                                    <textarea
                                        value={settings.titleInstructionVideo || `You are an expert in stock video SEO.
Analyze the provided video and identify:
- moving elements
- camera angle or motion
- scene type (interior/exterior)
- lighting and time of day

Generate ONE SEO-friendly video title that:
- starts with the main visual subject
- includes motion words only if visible
- avoids cinematic or narrative language
- stays under 75 characters
- follows Adobe Stock video title best practices`}
                                        onChange={(e) => setSettings({ ...settings, titleInstructionVideo: e.target.value })}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-64 resize-none font-mono"
                                        placeholder="Instruction for Video Title generation..."
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-2">SVG Title Instructions</label>
                                    <textarea
                                        value={settings.titleInstructionSvg || `You are a vector illustration SEO specialist.
Analyze the SVG/illustration and identify:
- main object or icon
- style (flat, outline, 3D, minimal)
- usage context (UI, web, app, print)
- color and theme

Generate ONE SEO-optimized title that:
- starts with the main object name
- includes "vector", "icon", or "illustration" if relevant
- is suitable for stock marketplaces
- avoids artistic language
- remains under 65 characters`}
                                        onChange={(e) => setSettings({ ...settings, titleInstructionSvg: e.target.value })}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-64 resize-none font-mono"
                                        placeholder="Instruction for SVG Title generation..."
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Image Description Instructions</label>
                                    <textarea
                                        value={settings.descriptionInstructionImage || (settings.descriptionInstruction || `Act as a top microstock contributor and SEO specialist.
Visually analyze the media and extract:
1. Primary subject (first phrase)
2. Supporting objects or details
3. Scene type (interior/exterior/isolated/background)
4. Lighting, colors, and style
5. Commercial usage intent

Write ONE description that:
- clearly describes only what is visible
- opens with the main subject in the first 5 words
- naturally includes common buyer search terms
- includes "copy space" or "background" ONLY if visible
- avoids emotional adjectives unless visually clear
- avoids filler words and repetition
- remains factual and professional
- is optimized for stock search engines
- is between 2–3 concise sentences`)}
                                        onChange={(e) => setSettings({ ...settings, descriptionInstructionImage: e.target.value })}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-64 resize-none font-mono"
                                        placeholder="Instruction for Image Description generation..."
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Video Description Instructions</label>
                                    <textarea
                                        value={settings.descriptionInstructionVideo || `You are an expert in stock video SEO.
Analyze the video and identify:
- moving subjects
- camera motion or angle
- environment and time of day
- lighting and scene style

Write ONE SEO-friendly description that:
- starts with the main moving subject
- describes visible motion accurately
- avoids cinematic or narrative language
- avoids assumptions about intent
- is suitable for Adobe Stock video listings
- uses 1–2 concise sentences`}
                                        onChange={(e) => setSettings({ ...settings, descriptionInstructionVideo: e.target.value })}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-64 resize-none font-mono"
                                        placeholder="Instruction for Video Description generation..."
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-2">SVG Description Instructions</label>
                                    <textarea
                                        value={settings.descriptionInstructionSvg || `You are a vector illustration SEO specialist.
Analyze the SVG/illustration and identify:
- main object or icon
- style (flat, outline, minimal, 3D)
- color palette
- usage context (web, app, UI, print)

Write ONE SEO-optimized description that:
- starts with the main object
- includes "vector", "icon", or "illustration" if relevant
- focuses on usability and clarity
- avoids artistic or emotional language
- is suitable for microstock platforms`}
                                        onChange={(e) => setSettings({ ...settings, descriptionInstructionSvg: e.target.value })}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-64 resize-none font-mono"
                                        placeholder="Instruction for SVG Description generation..."
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Keyword Generation Prompt (Advanced)</label>
                                    <textarea
                                        value={settings.keywordGenerationPrompt || DEFAULT_KEYWORD_GENERATION_PROMPT}
                                        onChange={(e) => setSettings({ ...settings, keywordGenerationPrompt: e.target.value })}
                                        className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-3 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-64 resize-none font-mono text-xs"
                                        placeholder="Full keyword generation prompt with SEO guidelines..."
                                    />
                                    <p className="text-xs text-slate-400 mt-2">Advanced: Full AI prompt for keyword generation. Use {'{{ keywordCount }}'} as placeholder for the keyword count setting.</p>
                                </div>
                            </div>
                        </div>

                        <div className="border-t border-slate-200 dark:border-slate-800 pt-4">
                            <h3 className="font-bold text-slate-900 dark:text-white">Adobe Stock Integration</h3>
                            <p className="text-sm text-slate-500 dark:text-slate-400 mb-4">Configure specific settings for Adobe Stock metadata.</p>
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-2">Category Instruction Prompt</label>
                                <textarea
                                    value={settings.adobeStockInstruction || DEFAULT_ADOBE_INSTRUCTION}
                                    onChange={(e) => setSettings({ ...settings, adobeStockInstruction: e.target.value })}
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl p-4 text-sm focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-900 outline-none transition-all h-32 resize-none font-mono"
                                    placeholder="Enter instructions for the AI to determine the category..."
                                />
                                <p className="text-xs text-slate-400 mt-2">This instruction is injected into the AI prompt when Adobe Stock is enabled.</p>
                            </div>
                        </div>
                    </div>
                )}
                <div className={`space-y-4 border-t border-b border-slate-200 dark:border-slate-800 py-6 ${configTab !== 'general' ? 'hidden' : ''}`}>
                    <div className="flex justify-between items-end mb-4">
                        <div>
                            <h3 className="font-bold text-slate-900 dark:text-white text-sm">Credits Per API Key (Model Allocation)</h3>
                            <p className="text-xs text-slate-500">Define credit limits for each model per valid key.</p>
                        </div>
                        <div className="text-right flex flex-col items-end gap-1">
                            <div className="flex items-center justify-end gap-2 mb-1">
                                <Clock size={10} className="text-slate-400" />
                                <span className="text-[10px] font-bold text-slate-500 uppercase">Resets in</span>
                                <span className="text-xs font-mono font-bold text-blue-600 dark:text-blue-400 bg-slate-100 dark:bg-slate-800 px-1.5 py-0.5 rounded">{timeToReset}</span>
                            </div>
                            <div className="flex items-center gap-2">
                                <span className="text-xs text-slate-500 uppercase font-bold">free TOTAL CREDITS</span>
                                <p className="text-xl font-bold text-amber-600 dark:text-amber-500">{totals.free.toLocaleString()}</p>
                            </div>
                            <div className="flex items-center gap-2">
                                <span className="text-xs text-slate-500 uppercase font-bold">paid total credits</span>
                                <p className="text-xl font-bold text-amber-600 dark:text-amber-500">{totals.paid.toLocaleString()}</p>
                            </div>
                        </div>
                    </div>

                    {['thinking', 'creative', 'video'].map(category => (
                        <div key={category} className="mb-6 last:mb-0">
                            <h4 className="text-xs font-bold text-slate-400 uppercase mb-3 flex items-center gap-2">
                                {category === 'thinking' && <BrainCircuit size={14} />}
                                {category === 'creative' && <Palette size={14} />}
                                {category === 'video' && <Video size={14} />}
                                {category.charAt(0).toUpperCase() + category.slice(1)} Models
                            </h4>
                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                                {MODELS.filter(m => m.category === category).map(model => {
                                    const settingKey = MODEL_TYPE_TO_SETTINGS[model.id];
                                    // FIXED: Fallback to defaults first, then 0, to ensure new models have correct initial values
                                    const defaultRef = DEFAULT_TIER_LIMITS[settingKey];
                                    const val = settings.modelLimits[settingKey] !== undefined ? settings.modelLimits[settingKey] : (defaultRef ? defaultRef.paid : 0);

                                    return (
                                        <div key={model.id} className="p-4 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl hover:border-amber-500/50 transition-colors">
                                            <div className="flex items-start justify-between gap-4">
                                                {/* LEFT SIDE: Model Info */}
                                                <div className="flex items-start gap-3 flex-1">
                                                    <div className={`w-1.5 h-12 rounded-full ${model.tagColor.split(' ')[0]}`}></div>
                                                    <div className="flex-1">
                                                        <p className="text-sm font-bold text-slate-900 dark:text-white mb-1">{model.name}</p>
                                                        <p className="text-xs text-slate-500 dark:text-slate-400">{model.desc}</p>
                                                    </div>
                                                </div>

                                                {/* RIGHT SIDE: Controls */}
                                                <div className="flex items-start gap-4">
                                                    {/* FREE/PAID Toggles */}
                                                    <div className="flex flex-col gap-2">
                                                        {/* FREE API Toggle */}
                                                        <button
                                                            onClick={() => {
                                                                if (settings) {
                                                                    const currentAccess = settings.modelAccessTiers || {};
                                                                    const modelAccess = currentAccess[model.id] || { free: true, paid: true };
                                                                    setSettings({
                                                                        ...settings,
                                                                        modelAccessTiers: {
                                                                            ...currentAccess,
                                                                            [model.id]: {
                                                                                ...modelAccess,
                                                                                free: !modelAccess.free
                                                                            }
                                                                        }
                                                                    });
                                                                }
                                                            }}
                                                            className={`px-3 py-1.5 rounded-lg text-xs font-bold uppercase tracking-wide transition ${(
                                                                settings.modelAccessTiers?.[model.id]?.free !== false)
                                                                ? 'bg-blue-500 text-white shadow-sm'
                                                                : 'bg-slate-200 text-slate-500 dark:bg-slate-800 dark:text-slate-600'
                                                                }`}
                                                            title={`FREE API: ${(settings.modelAccessTiers?.[model.id]?.free !== false) ? 'Enabled' : 'Disabled'}`}
                                                        >
                                                            🆓 FREE
                                                        </button>
                                                        {/* PAID API Toggle */}
                                                        <button
                                                            onClick={() => {
                                                                if (settings) {
                                                                    const currentAccess = settings.modelAccessTiers || {};
                                                                    const modelAccess = currentAccess[model.id] || { free: true, paid: true };
                                                                    setSettings({
                                                                        ...settings,
                                                                        modelAccessTiers: {
                                                                            ...currentAccess,
                                                                            [model.id]: {
                                                                                ...modelAccess,
                                                                                paid: !modelAccess.paid
                                                                            }
                                                                        }
                                                                    });
                                                                }
                                                            }}
                                                            className={`px-3 py-1.5 rounded-lg text-xs font-bold uppercase tracking-wide transition ${(
                                                                settings.modelAccessTiers?.[model.id]?.paid !== false)
                                                                ? 'bg-amber-500 text-white shadow-sm'
                                                                : 'bg-slate-200 text-slate-500 dark:bg-slate-800 dark:text-slate-600'
                                                                }`}
                                                            title={`PAID API: ${(settings.modelAccessTiers?.[model.id]?.paid !== false) ? 'Enabled' : 'Disabled'}`}
                                                        >
                                                            💎 PAID
                                                        </button>
                                                    </div>

                                                    {/* Credits Inputs */}
                                                    <div className="flex flex-col gap-2">
                                                        {/* FREE Credits */}
                                                        <div className="flex items-center gap-2">
                                                            <span className="text-xs font-semibold text-blue-600 dark:text-blue-400 w-12">FREE</span>
                                                            <input
                                                                type="number"
                                                                value={settings.modelLimitsByTier?.[settingKey]?.free ?? DEFAULT_TIER_LIMITS[settingKey]?.free ?? val}
                                                                onChange={(e) => {
                                                                    if (settings) {
                                                                        const currentLimits = settings.modelLimitsByTier || {};
                                                                        // Ensure we have a valid base object
                                                                        const defaultLimit = DEFAULT_TIER_LIMITS[settingKey] || { free: 0, paid: 0 };
                                                                        const modelLimits = currentLimits[settingKey] || { free: defaultLimit.free, paid: defaultLimit.paid };

                                                                        setSettings({
                                                                            ...settings,
                                                                            modelLimitsByTier: {
                                                                                ...currentLimits,
                                                                                [settingKey]: {
                                                                                    ...modelLimits,
                                                                                    free: Number(e.target.value)
                                                                                }
                                                                            }
                                                                        });
                                                                    }
                                                                }}
                                                                className="w-20 bg-blue-50 dark:bg-blue-950/30 border-2 border-blue-300 dark:border-blue-700 rounded-lg px-3 py-1.5 text-sm font-bold text-center text-blue-900 dark:text-blue-100 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 dark:focus:ring-blue-800 outline-none transition-all"
                                                            />
                                                        </div>
                                                        {/* PAID Credits */}
                                                        <div className="flex items-center gap-2">
                                                            <span className="text-xs font-semibold text-amber-600 dark:text-amber-400 w-12">PAID</span>
                                                            <input
                                                                type="number"
                                                                value={settings.modelLimitsByTier?.[settingKey]?.paid ?? DEFAULT_TIER_LIMITS[settingKey]?.paid ?? val}
                                                                onChange={(e) => {
                                                                    if (settings) {
                                                                        const currentLimits = settings.modelLimitsByTier || {};
                                                                        // Ensure we have a valid base object
                                                                        const defaultLimit = DEFAULT_TIER_LIMITS[settingKey] || { free: 0, paid: 0 };
                                                                        const modelLimits = currentLimits[settingKey] || { free: defaultLimit.free, paid: defaultLimit.paid };

                                                                        setSettings({
                                                                            ...settings,
                                                                            modelLimitsByTier: {
                                                                                ...currentLimits,
                                                                                [settingKey]: {
                                                                                    ...modelLimits,
                                                                                    paid: Number(e.target.value)
                                                                                }
                                                                            }
                                                                        });
                                                                    }
                                                                }}
                                                                className="w-20 bg-amber-50 dark:bg-amber-950/30 border-2 border-amber-300 dark:border-amber-700 rounded-lg px-3 py-1.5 text-sm font-bold text-center text-amber-900 dark:text-amber-100 focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-800 outline-none transition-all"
                                                            />
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            {/* Model Code - Below */}
                                            <div className="mt-3 pt-3 border-t border-slate-200 dark:border-slate-800">
                                                <div className="flex items-center gap-3">
                                                    <span className="text-xs font-semibold text-slate-500 dark:text-slate-400 whitespace-nowrap">MODEL CODE</span>
                                                    <input
                                                        type="text"
                                                        value={settings.modelIds?.[settingKey] || ''}
                                                        onChange={(e) => {
                                                            if (settings) {
                                                                setSettings({
                                                                    ...settings,
                                                                    modelIds: {
                                                                        ...settings.modelIds,
                                                                        [settingKey]: e.target.value
                                                                    }
                                                                });
                                                            }
                                                        }}
                                                        placeholder={model.id}
                                                        className="flex-1 bg-slate-100 dark:bg-slate-900 border border-slate-300 dark:border-slate-700 rounded-lg px-3 py-1.5 text-xs font-mono text-slate-700 dark:text-slate-300 focus:border-amber-500 focus:ring-2 focus:ring-amber-200 dark:focus:ring-amber-800 outline-none transition-all"
                                                    />
                                                </div>
                                            </div>
                                        </div>
                                    );
                                })}
                            </div>
                        </div>
                    ))}<div className="h-4"></div>
                </div>
                <div className={`space-y-3 ${configTab !== 'general' ? 'hidden' : ''}`}><h3 className="font-bold text-slate-900 dark:text-white text-sm flex items-center gap-2">App Static API Key <span className="px-2 py-0.5 bg-amber-100 dark:bg-amber-900/30 text-amber-600 dark:text-amber-400 text-[10px] rounded-full">SYSTEM</span><div className="flex items-center gap-1 bg-red-500 text-white text-[10px] font-bold rounded-md shadow-sm px-2 py-0.5"><span>Used: {settings.systemApiUsage?.toLocaleString() || 0}</span><button onClick={() => setSettings({ ...settings, systemApiUsage: 0 })} className="hover:text-red-200 transition ml-1" title="Reset Counter"><RotateCcw size={10} /></button></div></h3><input type="password" value={settings.appStaticApiKey || ''} onChange={(e) => setSettings({ ...settings, appStaticApiKey: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-700 dark:text-slate-300 text-sm focus:border-amber-500 outline-none" placeholder="AIza..." /><p className="text-xs text-slate-500">Master key used for generating monthly trends info for user dashboards.</p></div>
                <div className={`space-y-3 ${configTab !== 'general' ? 'hidden' : ''}`}><h3 className="font-bold text-slate-900 dark:text-white text-sm">Maintenance Message</h3><textarea value={settings.maintenanceMessage} onChange={(e) => setSettings({ ...settings, maintenanceMessage: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-700 dark:text-slate-300 text-sm focus:border-amber-500 outline-none" /></div>
                <div className={`grid grid-cols-1 md:grid-cols-2 gap-6 ${configTab !== 'general' ? 'hidden' : ''}`}><div className="space-y-3"><h3 className="font-bold text-slate-900 dark:text-white text-sm">Maintenance Bypass Pre-Code</h3><input type="text" value={settings.maintenancePreCode || ''} onChange={(e) => setSettings({ ...settings, maintenancePreCode: e.target.value })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-700 dark:text-slate-300 text-sm focus:border-amber-500 outline-none" placeholder="e.g., admin123" /><p className="text-xs text-slate-500">Code required to access login page during maintenance.</p></div><div className="space-y-3"><h3 className="font-bold text-slate-900 dark:text-white text-sm">Maintenance End Time</h3><input type="datetime-local" value={settings.maintenanceEndTime ? new Date(settings.maintenanceEndTime - new Date().getTimezoneOffset() * 60000).toISOString().slice(0, 16) : ''} onChange={(e) => { const date = new Date(e.target.value); setSettings({ ...settings, maintenanceEndTime: date.getTime() }); }} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-slate-700 dark:text-slate-300 text-sm focus:border-amber-500 outline-none" /><p className="text-xs text-slate-500">Set the target time for the countdown timer.</p></div></div>

                {/* Sidebar Management */}
                <div className={`bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-8 space-y-6 max-w-6xl shadow-sm dark:shadow-none ${configTab !== 'general' ? 'hidden' : ''}`}>
                    <h3 className="font-bold text-slate-900 dark:text-white flex items-center gap-2"><LayoutTemplate className="text-amber-500" /> Sidebar Management</h3>
                    <p className="text-sm text-slate-500 dark:text-slate-400">Control visibility and status of sidebar items.</p>

                    <div className="space-y-6">
                        {[
                            {
                                title: 'Main',
                                items: [
                                    { id: 'DASHBOARD', label: 'Dashboard', icon: LayoutDashboard },
                                    { id: 'CHAT_INTELLIGENCE', label: 'Market Intelligence', icon: Sparkles },
                                    { id: 'MEDIA_ANALYSIS', label: 'Metadata AI', icon: ScanEye },
                                ]
                            },
                            {
                                title: 'Creation Tools',
                                items: [
                                    { id: 'IMAGE_GENERATION', label: 'Image Generation', icon: ImageIcon },
                                    { id: 'IMAGE_EDITOR', label: 'Image Editor', icon: PencilRuler },
                                    { id: 'VIDEO_GENERATION', label: 'Video Generation', icon: Video },
                                    { id: 'REVIEWER_AI', label: 'Reviewer AI', icon: ScanEye },
                                    { id: 'PROMPT_GENERATOR', label: 'Auto-Prompt Gen', icon: Sparkles },
                                ]
                            },
                            {
                                title: 'Management',
                                items: [
                                    { id: 'LIBRARY', label: 'My Library', icon: Folder },
                                    { id: 'FAVORITES', label: 'Favorites', icon: Star },
                                    { id: 'TRASH', label: 'Trash', icon: Trash2 },
                                ]
                            }
                        ].map((group, idx) => (
                            <div key={idx} className="space-y-3">
                                <h4 className="text-xs font-bold text-slate-500 uppercase tracking-wider px-1">{group.title}</h4>
                                <div className="grid grid-cols-1 gap-3">
                                    {group.items.map((item) => {
                                        const setting = settings.sidebarSettings?.[item.id] || { id: item.id, enabled: true, comingSoon: false };

                                        return (
                                            <div key={item.id} className="flex items-center justify-between p-4 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-800 rounded-xl hover:border-slate-300 dark:hover:border-slate-700 transition-colors">
                                                <div className="flex items-center gap-3">
                                                    <div className="p-2 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-lg text-slate-500">
                                                        <item.icon size={18} />
                                                    </div>
                                                    <div>
                                                        <h4 className="font-bold text-sm text-slate-900 dark:text-white">{item.label}</h4>
                                                        <div className="flex gap-2 mt-1">
                                                            <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold uppercase ${setting.enabled ? 'bg-green-100 text-green-600 dark:bg-green-900/30 dark:text-green-400' : 'bg-red-100 text-red-600 dark:bg-red-900/30 dark:text-red-400'}`}>
                                                                {setting.enabled ? 'Visible' : 'Hidden'}
                                                            </span>
                                                            {setting.comingSoon && <span className="text-[10px] px-2 py-0.5 rounded-full font-bold uppercase bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400">Coming Soon</span>}
                                                        </div>
                                                    </div>
                                                </div>
                                                <div className="flex items-center gap-6">
                                                    <div className="flex flex-col items-end gap-1">
                                                        <span className="text-[10px] font-bold text-slate-500 uppercase">Enabled</span>
                                                        <Toggle checked={setting.enabled} onChange={() => handleSidebarToggle(item.id, 'enabled')} />
                                                    </div>
                                                    <div className="flex flex-col items-end gap-1">
                                                        <span className="text-[10px] font-bold text-slate-500 uppercase">Coming Soon</span>
                                                        <Toggle checked={setting.comingSoon} onChange={() => handleSidebarToggle(item.id, 'comingSoon')} />
                                                    </div>
                                                </div>
                                            </div>
                                        );
                                    })}
                                </div>
                            </div>
                        ))}
                    </div>
                </div>

                {/* AI Reviewer Configuration */}
                <div className={`bg-gradient-to-r from-sky-50 to-indigo-50 dark:from-slate-900 dark:to-indigo-950 border-2 border-sky-200 dark:border-indigo-900 rounded-3xl p-8 shadow-sm hover:border-sky-300 dark:hover:border-indigo-800 transition-colors ${configTab !== 'reviewer' ? 'hidden' : ''}`}>
                    <div className="flex items-center gap-3 mb-6">
                        <div className="p-3 bg-sky-500/10 rounded-xl text-sky-600 dark:text-sky-400">
                            <Camera size={24} />
                        </div>
                        <div>
                            <h3 className="font-bold text-lg text-slate-900 dark:text-white flex items-center gap-2">
                                AI Reviewer Configuration
                                <span className="text-xs px-2 py-1 bg-amber-500 text-white rounded font-bold uppercase">Admin Only</span>
                            </h3>
                            <p className="text-xs text-slate-500">Customize system prompts for image review two-step verification.</p>
                        </div>
                    </div>

                    <div className="space-y-6">
                        <div>
                            <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-2 block">
                                System Prompt for API Call A (Human/Anatomy Check)
                            </label>
                            <textarea
                                value={anatomyPrompt}
                                onChange={(e) => {
                                    setAnatomyPrompt(e.target.value);
                                    localStorage.setItem('sf_reviewer_prompt_anatomy_v3', e.target.value);
                                }}
                                placeholder="Leave empty to use default prompt. Customize the prompt used to detect human anatomical errors..."
                                className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-4 text-sm font-mono text-slate-900 dark:text-white outline-none focus:border-sky-500 focus:ring-2 focus:ring-sky-500/20 transition min-h-[200px] resize-y"
                            />
                            <p className="text-xs text-slate-500 mt-2">
                                This prompt analyzes images for human anatomical defects. Returns score 0 (defects found) or 100 (no defects).
                            </p>
                        </div>

                        <div>
                            <label className="text-sm font-bold text-slate-700 dark:text-slate-300 mb-2 block">
                                System Prompt for API Call B (Stock Quality Check)
                            </label>
                            <textarea
                                value={qualityPrompt}
                                onChange={(e) => {
                                    setQualityPrompt(e.target.value);
                                    localStorage.setItem('sf_reviewer_prompt_quality_v3', e.target.value);
                                }}
                                placeholder="Leave empty to use default prompt. Customize the prompt used for technical quality assessment..."
                                className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl p-4 text-sm font-mono text-slate-900 dark:text-white outline-none focus:border-sky-500 focus:ring-2 focus:ring-sky-500/20 transition min-h-[200px] resize-y"
                            />
                            <p className="text-xs text-slate-500 mt-2">
                                This prompt analyzes technical quality (noise, lighting, composition, etc.). Returns score 0-100.
                            </p>
                        </div>

                        <button
                            onClick={() => {
                                setAnatomyPrompt(DEFAULT_ANATOMY_PROMPT);
                                setQualityPrompt(DEFAULT_QUALITY_PROMPT);
                                localStorage.removeItem('sf_reviewer_prompt_anatomy_v3');
                                localStorage.removeItem('sf_reviewer_prompt_quality_v3');
                            }}
                            className="px-6 py-2 bg-slate-200 dark:bg-slate-800 hover:bg-slate-300 dark:hover:bg-slate-700 text-slate-700 dark:text-slate-200 rounded-lg font-bold text-sm transition"
                        >
                            Reset to Defaults
                        </button>

                        {/* Scoring Rules Display */}
                        <div className="mt-8 pt-6 border-t border-slate-200 dark:border-slate-700">
                            <div className="flex items-center justify-between mb-4">
                                <h4 className="text-sm font-bold text-slate-700 dark:text-slate-300">AI Reviewer Scoring Rules</h4>
                                <span className="text-xs px-2 py-1 bg-red-500 text-white rounded font-bold uppercase">Editable</span>
                            </div>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                {/* Anatomy Check Rule */}
                                <div className="bg-amber-50 dark:bg-amber-900/10 border-2 border-red-500 dark:border-red-600 rounded-xl p-4">
                                    <div className="flex items-center gap-2 mb-3">
                                        <div className="w-2 h-2 bg-amber-500 rounded-full"></div>
                                        <h5 className="font-bold text-slate-900 dark:text-white text-sm">Anatomy Check</h5>
                                    </div>

                                    {/* Pass Score Input */}
                                    <div className="mb-3">
                                        <span className="text-xs text-slate-600 dark:text-slate-400">= Accepted (No defects)</span>
                                    </div>
                                </div>

                                {/* Fail Score Input */}
                                <div>
                                    <label className="text-[10px] text-slate-600 dark:text-slate-400 font-bold mb-1 block">Fail Score (Defects Found)</label>
                                    <div className="flex items-center gap-2">
                                        <input
                                            type="number"
                                            min="0"
                                            max="100"
                                            value={anatomyFailScore}
                                            onChange={(e) => {
                                                const val = Number(e.target.value);
                                                setAnatomyFailScore(val);
                                                localStorage.setItem('sf_reviewer_anatomy_fail', val.toString());
                                            }}
                                            className="w-20 bg-white dark:bg-slate-900 border border-red-400 dark:border-red-600 rounded-lg px-2 py-1 text-sm font-bold text-red-600 dark:text-red-400 focus:border-red-500 outline-none"
                                        />
                                        <span className="text-xs text-slate-600 dark:text-slate-400">= Rejected (Defects found)</span>
                                    </div>
                                </div>

                                {/* Quality Check Rule */}
                                <div className="bg-amber-50 dark:bg-amber-900/10 border-2 border-red-500 dark:border-red-600 rounded-xl p-4">
                                    <div className="flex items-center gap-2 mb-3">
                                        <div className="w-2 h-2 bg-amber-500 rounded-full"></div>
                                        <h5 className="font-bold text-slate-900 dark:text-white text-sm">Quality Check</h5>
                                    </div>

                                    {/* Minimum Quality Score Input */}
                                    <div>
                                        <label className="text-[10px] text-slate-600 dark:text-slate-400 font-bold mb-1 block">Minimum Passing Score</label>
                                        <div className="flex items-center gap-2 mb-2">
                                            <input
                                                type="number"
                                                min="0"
                                                max="100"
                                                value={qualityMinScore}
                                                onChange={(e) => {
                                                    const val = Number(e.target.value);
                                                    setQualityMinScore(val);
                                                    localStorage.setItem('sf_reviewer_quality_min', val.toString());
                                                    // Auto-update total threshold based on quality min
                                                    const newThreshold = anatomyPassScore + val;
                                                    setTotalScoreThreshold(newThreshold);
                                                    localStorage.setItem('sf_reviewer_total_threshold', newThreshold.toString());
                                                }}
                                                className="w-20 bg-white dark:bg-slate-900 border border-amber-400 dark:border-amber-600 rounded-lg px-2 py-1 text-sm font-bold text-amber-600 dark:text-amber-400 focus:border-amber-500 outline-none"
                                            />
                                        </div>
                                    </div>

                                    <p className="text-xs text-slate-600 dark:text-slate-400 mb-2">
                                        <span className="font-bold text-red-600 dark:text-red-400">0-{qualityMinScore - 1}</span> = Rejected (Poor quality)
                                    </p>
                                    <p className="text-xs text-slate-600 dark:text-slate-400">
                                        <span className="font-bold text-green-600 dark:text-green-400">{qualityMinScore}-100</span> = Can be Accepted
                                    </p>
                                    <p className="text-[10px] text-slate-500 dark:text-slate-500 mt-2 italic">
                                        Technical quality score (noise, blur, lighting, etc.)
                                    </p>
                                </div>
                            </div>

                            {/* Total Score Info */}
                            <div className="mt-4 bg-blue-50 dark:bg-blue-900/10 border-2 border-red-500 dark:border-red-600 rounded-xl p-4">
                                <h5 className="font-bold text-slate-900 dark:text-white text-sm mb-3">Final Acceptance Rule</h5>

                                {/* Total Threshold Input */}
                                <div className="mb-3">
                                    <label className="text-[10px] text-slate-600 dark:text-slate-400 font-bold mb-1 block">Total Score Threshold</label>
                                    <div className="flex items-center gap-2">
                                        <input
                                            type="number"
                                            min="0"
                                            max="200"
                                            value={totalScoreThreshold}
                                            onChange={(e) => {
                                                const val = Number(e.target.value);
                                                setTotalScoreThreshold(val);
                                                localStorage.setItem('sf_reviewer_total_threshold', val.toString());
                                            }}
                                            className="w-20 bg-white dark:bg-slate-900 border border-blue-400 dark:border-blue-600 rounded-lg px-2 py-1 text-sm font-bold text-blue-600 dark:text-blue-400 focus:border-blue-500 outline-none"
                                        />
                                        <span className="text-xs text-slate-600 dark:text-slate-400">out of 200 points</span>
                                    </div>
                                </div>

                                <p className="text-xs text-slate-600 dark:text-slate-400">
                                    Total Score = Anatomy (0-100) + Quality (0-100) = Max 200 points
                                </p>
                                <p className="text-xs text-slate-600 dark:text-slate-400 mt-1">
                                    <span className="font-bold text-green-600 dark:text-green-400">≥ {totalScoreThreshold} points</span> = ACCEPTED
                                    <span className="mx-2">•</span>
                                    <span className="font-bold text-red-600 dark:text-red-400">&lt; {totalScoreThreshold} points</span> = REJECTED
                                </p>
                                <p className="text-[10px] text-slate-500 dark:text-slate-500 mt-2">
                                    Example: Anatomy {anatomyPassScore} + Quality {qualityMinScore - 1} = {anatomyPassScore + qualityMinScore - 1} → <span className="text-red-600 dark:text-red-400 font-bold">REJECTED</span>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <button onClick={handleSave} disabled={isSaving} className="px-6 py-2 bg-amber-500 text-slate-950 font-bold rounded-lg hover:bg-amber-600 transition disabled:opacity-50">{isSaving ? 'Saving...' : 'Save Changes'}</button>
            </div>
        </div>
    );
};

const LandingPageEditor = () => {
    const [content, setContent] = useState<any>(null);
    const [activeTab, setActiveTab] = useState<'general' | 'hero' | 'features' | 'impact' | 'about' | 'pricing' | 'auth'>('general');
    const [isSaving, setIsSaving] = useState(false);
    const [newSlideUrl, setNewSlideUrl] = useState('');
    const [newSlideTitle, setNewSlideTitle] = useState('');
    const [newSlideType, setNewSlideType] = useState<'image' | 'video'>('image');

    useEffect(() => { adminService.getLandingContent().then(setContent); }, []);

    const handleSave = async () => { setIsSaving(true); await adminService.saveLandingContent(content); setIsSaving(false); alert("Landing page updated successfully!"); };
    const handleToggle = (path: string[], key: string) => { const newContent = { ...content }; let current = newContent; for (let i = 0; i < path.length; i++) { current = current[path[i]]; } current[key] = !current[key]; setContent(newContent); };
    const handleAddSlide = () => { if (!newSlideUrl) return; const slides = content.globalImpact?.slides || []; if (slides.length >= 20) { alert("Maximum 20 slides allowed."); return; } const newSlide = { id: Math.random().toString(36).substring(2, 9), type: newSlideType, url: newSlideUrl, title: newSlideTitle || 'Untitled Slide' }; const newImpact = { ...content.globalImpact, slides: [...slides, newSlide] }; setContent({ ...content, globalImpact: newImpact }); setNewSlideUrl(''); setNewSlideTitle(''); };
    const handleDeleteSlide = (id: string) => { const slides = content.globalImpact?.slides || []; const newSlides = slides.filter((s: any) => s.id !== id); setContent({ ...content, globalImpact: { ...content.globalImpact, slides: newSlides } }); };
    const handleUpdateSlide = (id: string, field: string, value: string) => { const slides = content.globalImpact?.slides || []; const newSlides = slides.map((s: any) => s.id === id ? { ...s, [field]: value } : s); setContent({ ...content, globalImpact: { ...content.globalImpact, slides: newSlides } }); };

    const InputGroup = ({ label, value, onChange }: { label: string, value: string, onChange: (v: string) => void }) => (<div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">{label}</label><input type="text" value={value || ''} onChange={(e) => onChange(e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm text-slate-900 dark:text-white focus:border-amber-500 outline-none transition-colors" /></div>);
    const TextAreaGroup = ({ label, value, onChange }: { label: string, value: string, onChange: (v: string) => void }) => (<div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">{label}</label><textarea value={value || ''} onChange={(e) => onChange(e.target.value)} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm text-slate-900 dark:text-white focus:border-amber-500 outline-none h-24 resize-none leading-relaxed transition-colors" /></div>);

    if (!content) return <div className="p-10 text-center"><Loader2 className="animate-spin mx-auto" /></div>;

    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <div>

                    <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><LayoutTemplate className="text-amber-500" /> Landing Page Editor</h2>
                    <p className="text-slate-500 dark:text-slate-400">Customize the public facing content.</p>
                </div>
                <button onClick={handleSave} disabled={isSaving} className="px-6 py-2 bg-amber-500 hover:bg-amber-600 text-slate-950 font-bold rounded-xl flex items-center gap-2 transition disabled:opacity-50">
                    {isSaving ? <Loader2 className="animate-spin" size={18} /> : <Save size={18} />} Save Changes
                </button>
            </div>

            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden shadow-sm dark:shadow-none">
                <div className="flex border-b border-slate-200 dark:border-slate-800 overflow-x-auto">
                    {['general', 'hero', 'auth', 'features', 'impact', 'about', 'pricing'].map(tab => (
                        <button key={tab} onClick={() => setActiveTab(tab as any)} className={`px-6 py-4 text-sm font-bold capitalize transition-colors whitespace-nowrap ${activeTab === tab ? 'bg-slate-100 dark:bg-slate-800 text-slate-900 dark:text-white border-b-2 border-amber-500' : 'text-slate-500 hover:text-slate-800 dark:hover:text-slate-300'} `}>
                            {tab === 'impact' ? 'Impact Slider' : tab === 'pricing' ? 'Pricing & CTA' : tab === 'auth' ? 'Auth Screen' : tab}
                        </button>
                    ))}
                </div>

                <div className="p-8 space-y-6">
                    {activeTab === 'general' && (
                        <div className="space-y-6 max-w-3xl">
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Global Settings</h3>
                            <InputGroup label="Application Name" value={content.general?.appName || 'StockForge AI'} onChange={(v) => setContent({ ...content, general: { ...content.general, appName: v } })} />
                            <InputGroup label="App Logo URL" value={content.general?.appLogo || ''} onChange={(v) => setContent({ ...content, general: { ...content.general, appLogo: v } })} />
                            <InputGroup label="Footer Copyright Text" value={content.general?.footerText || '© 2024 StockForge AI. Made with 💛 and Gemini.'} onChange={(v) => setContent({ ...content, general: { ...content.general, footerText: v } })} />
                            <div className="h-px bg-slate-200 dark:bg-slate-800 my-4"></div>
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Navigation Menu Labels</h3>
                            <div className="grid grid-cols-3 gap-4">
                                <InputGroup label="Features Link" value={content.navigation?.featuresLabel || 'Features'} onChange={(v) => setContent({ ...content, navigation: { ...content.navigation, featuresLabel: v } })} />
                                <InputGroup label="Pricing Link" value={content.navigation?.pricingLabel || 'Pricing'} onChange={(v) => setContent({ ...content, navigation: { ...content.navigation, pricingLabel: v } })} />
                                <InputGroup label="About Link" value={content.navigation?.aboutLabel || 'About'} onChange={(v) => setContent({ ...content, navigation: { ...content.navigation, aboutLabel: v } })} />
                            </div>
                        </div>
                    )}
                    {activeTab === 'hero' && (
                        <div className="space-y-6 max-w-3xl">
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Hero Section</h3>
                            <InputGroup label="Hero Title" value={content.hero?.title || ''} onChange={(v) => setContent({ ...content, hero: { ...content.hero, title: v } })} />
                            <TextAreaGroup label="Hero Subtitle" value={content.hero?.subtitle || ''} onChange={(v) => setContent({ ...content, hero: { ...content.hero, subtitle: v } })} />
                            <InputGroup label="CTA Button Text" value={content.hero?.ctaText || ''} onChange={(v) => setContent({ ...content, hero: { ...content.hero, ctaText: v } })} />
                            <InputGroup label="CTA Button Link" value={content.hero?.ctaLink || ''} onChange={(v) => setContent({ ...content, hero: { ...content.hero, ctaLink: v } })} />
                        </div>
                    )}
                    {activeTab === 'auth' && (
                        <div className="space-y-6 max-w-3xl">
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Auth Screen</h3>
                            <InputGroup label="Login Title" value={content.auth?.loginTitle || ''} onChange={(v) => setContent({ ...content, auth: { ...content.auth, loginTitle: v } })} />
                            <TextAreaGroup label="Login Subtitle" value={content.auth?.loginSubtitle || ''} onChange={(v) => setContent({ ...content, auth: { ...content.auth, loginSubtitle: v } })} />
                            <InputGroup label="Register Title" value={content.auth?.registerTitle || ''} onChange={(v) => setContent({ ...content, auth: { ...content.auth, registerTitle: v } })} />
                            <TextAreaGroup label="Register Subtitle" value={content.auth?.registerSubtitle || ''} onChange={(v) => setContent({ ...content, auth: { ...content.auth, registerSubtitle: v } })} />
                        </div>
                    )}
                    {activeTab === 'features' && (
                        <div className="space-y-6 max-w-3xl">
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Features Section</h3>
                            <InputGroup label="Section Title" value={content.features?.title || ''} onChange={(v) => setContent({ ...content, features: { ...content.features, title: v } })} />
                            <TextAreaGroup label="Section Description" value={content.features?.description || ''} onChange={(v) => setContent({ ...content, features: { ...content.features, description: v } })} />
                        </div>
                    )}
                    {activeTab === 'impact' && (
                        <div className="space-y-6">
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Global Impact Slider</h3>
                            <p className="text-sm text-slate-500 dark:text-slate-400 mb-4">Add up to 20 slides showcasing your global impact.</p>
                            <div className="flex gap-3 mb-4">
                                <input type="text" placeholder="Slide URL (image or video)" value={newSlideUrl} onChange={(e) => setNewSlideUrl(e.target.value)} className="flex-1 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm" />
                                <input type="text" placeholder="Slide Title" value={newSlideTitle} onChange={(e) => setNewSlideTitle(e.target.value)} className="flex-1 bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm" />
                                <select value={newSlideType} onChange={(e) => setNewSlideType(e.target.value as 'image' | 'video')} className="bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-2.5 text-sm">
                                    <option value="image">Image</option>
                                    <option value="video">Video</option>
                                </select>
                                <button onClick={handleAddSlide} className="px-4 py-2 bg-amber-500 text-slate-950 font-bold rounded-lg hover:bg-amber-600 transition">Add Slide</button>
                            </div>
                            <div className="space-y-2">
                                {content.globalImpact?.slides?.map((slide: any) => (
                                    <div key={slide.id} className="flex items-center gap-3 p-3 bg-slate-50 dark:bg-slate-950 rounded-lg border border-slate-200 dark:border-slate-800">
                                        <span className="text-xs text-slate-500">{slide.type}</span>
                                        <input type="text" value={slide.title} onChange={(e) => handleUpdateSlide(slide.id, 'title', e.target.value)} className="flex-1 bg-transparent border-none text-sm text-slate-900 dark:text-white" />
                                        <input type="text" value={slide.url} onChange={(e) => handleUpdateSlide(slide.id, 'url', e.target.value)} className="flex-1 bg-transparent border-none text-sm text-slate-500" />
                                        <DoubleConfirmDeleteButton onDelete={() => handleDeleteSlide(slide.id)} />
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                    {activeTab === 'about' && (
                        <div className="space-y-6 max-w-3xl">
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">About Section</h3>
                            <InputGroup label="Section Title" value={content.about?.title || ''} onChange={(v) => setContent({ ...content, about: { ...content.about, title: v } })} />
                            <TextAreaGroup label="Section Content" value={content.about?.content || ''} onChange={(v) => setContent({ ...content, about: { ...content.about, content: v } })} />
                        </div>
                    )}
                    {activeTab === 'pricing' && (
                        <div className="space-y-6 max-w-3xl">
                            <h3 className="font-bold text-slate-900 dark:text-white text-lg mb-2">Pricing & CTA Section</h3>
                            <InputGroup label="Section Title" value={content.pricing?.title || ''} onChange={(v) => setContent({ ...content, pricing: { ...content.pricing, title: v } })} />
                            <TextAreaGroup label="Section Description" value={content.pricing?.description || ''} onChange={(v) => setContent({ ...content, pricing: { ...content.pricing, description: v } })} />
                            <InputGroup label="CTA Button Text" value={content.pricing?.ctaText || ''} onChange={(v) => setContent({ ...content, pricing: { ...content.pricing, ctaText: v } })} />
                        </div>
                    )}
                </div>
            </div>
        </div>
    );

};

const UsersManagement = ({ onMessageUser }: { onMessageUser: (uid: string) => void }) => {
    const [users, setUsers] = useState<User[]>([]);
    const [search, setSearch] = useState('');
    const [viewMode, setViewMode] = useState<'active' | 'deleted'>('active');
    const [selectedUser, setSelectedUser] = useState<User | null>(null);
    const [notification, setNotification] = useState<{ message: string, type: 'success' | 'error' | 'warning' } | null>(null);


    const showNotification = (message: string, type: 'success' | 'error' | 'warning' = 'success') => {
        setNotification({ message, type });
        setTimeout(() => setNotification(null), 5000);
    };

    useEffect(() => { loadUsers(); }, []);
    const loadUsers = async () => {
        try {
            console.log("Loading users...");
            const data = await adminService.getUsers();
            console.log("Loaded users:", data.length);
            console.log("Loaded users:", data.length);
            setUsers(data); // Keep all (filtered in render)
        } catch (e) {
            console.error("Failed to load users:", e);
        }
    };
    const handleToggleBan = async (uid: string) => {
        console.log("Toggling ban for:", uid);
        try {
            await adminService.toggleUserBan(uid);
            console.log("Ban toggled successfully");
            await loadUsers();
        } catch (error) {
            console.error("Failed to toggle ban:", error);
            alert("Failed to ban/unban user: " + (error as any).message);
        }
    };
    const handleRoleChange = async (uid: string, currentRole: string) => {
        const newRole = currentRole === 'admin' ? 'user' : 'admin';
        console.log("Changing role for:", uid, "to", newRole);
        try {
            await adminService.updateUserRole(uid, newRole as any);
            console.log("Role changed successfully");
            await loadUsers();
        } catch (error) {
            console.error("Failed to change role:", error);
            alert("Failed to change role: " + (error as any).message);
        }
    };
    const handleDeleteUser = async (uid: string, role: string) => {
        if (role === 'admin') {
            showNotification("CRITICAL SECURITY WARNING: You cannot delete an Administrator account. Demote this user first if you wish to remove them.", 'warning');
            return;
        }
        console.log("Attempting delete for:", uid);
        try {
            await adminService.deleteUser(uid);
            console.log("User deleted successfully");
            await loadUsers();
        } catch (error) {
            console.error("Delete failed", error);
            alert("Failed to delete user: " + (error as any).message);
        }
    };

    const handleRestoreUser = async (uid: string) => {
        try {
            await adminService.restoreUser(uid);
            await loadUsers();
            showNotification("User restored successfully");
        } catch (error) {
            showNotification("Restore failed", 'error');
        }
    };

    const handlePermanentDelete = async (uid: string) => {
        const user = users.find(u => u.uid === uid);
        const userName = user ? user.displayName : 'User';

        if (!confirm(`ARE YOU SURE? This will permanently erase ${userName} and ALL their data (Assets, API keys, etc). This cannot be undone.`)) return;

        try {
            await adminService.permanentDeleteUser(uid);
            await loadUsers();
            showNotification(`${userName} PERMANENTLY deleted`, 'warning');
        } catch (error) {
            showNotification("Permanent delete failed", 'error');
        }
    };
    const handleMessageClick = (uid: string) => {
        console.log("Message clicked for:", uid);
        if (onMessageUser) {
            onMessageUser(uid);
        } else {
            console.error("onMessageUser prop is missing!");
        }
    };

    const filteredUsers = users.filter(u => {
        const matchesSearch = (u.displayName || '').toLowerCase().includes(search.toLowerCase()) || (u.email || '').toLowerCase().includes(search.toLowerCase());
        const matchesView = viewMode === 'active' ? !u.isDeleted : u.isDeleted;
        return matchesSearch && matchesView;
    });

    return (
        <div className="space-y-6">
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2">
                    <Users className="text-blue-500" /> User Management
                </h2>

                <div className="flex bg-slate-100 dark:bg-slate-900 p-1 rounded-lg border border-slate-200 dark:border-slate-800">
                    <button onClick={() => setViewMode('active')} className={`px-4 py-1.5 rounded-md text-sm font-bold transition ${viewMode === 'active' ? 'bg-white dark:bg-slate-800 shadow text-blue-600' : 'text-slate-500'}`}>Active Users</button>
                    <button onClick={() => setViewMode('deleted')} className={`px-4 py-1.5 rounded-md text-sm font-bold transition ${viewMode === 'deleted' ? 'bg-white dark:bg-slate-800 shadow text-red-600' : 'text-slate-500'}`}>Bin / Deleted</button>
                </div>
            </div>

            <div className="relative">
                <div className="absolute inset-y-0 left-3 flex items-center pointer-events-none">
                    <Search className="text-slate-400" size={18} />
                </div>
                <input
                    type="text"
                    value={search}
                    onChange={e => setSearch(e.target.value)}
                    className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl py-3 pl-10 pr-4 text-sm outline-none focus:border-blue-500 transition-colors"
                    placeholder="Search users by name or email..."
                />
            </div>
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden shadow-sm"><table className="w-full text-sm text-left"><thead className="bg-slate-50 dark:bg-slate-950 text-slate-500 border-b border-slate-200 dark:border-slate-800"><tr><th className="p-4 font-bold">User</th><th className="p-4 font-bold">Role</th><th className="p-4 font-bold">Status</th><th className="p-4 font-bold">IP / Country</th><th className="p-4 font-bold">Joined</th><th className="p-4 font-bold text-right">Actions</th></tr></thead><tbody className="divide-y divide-slate-100 dark:divide-slate-800">{filteredUsers.map(user => (<tr key={user.uid} className="hover:bg-slate-50 dark:hover:bg-slate-800/50 transition"><td className="p-4"><div className="flex items-center gap-3"><div className="w-8 h-8 rounded-full bg-slate-200 dark:bg-slate-800 flex items-center justify-center overflow-hidden">{user.photoURL ? <img src={user.photoURL} className="w-full h-full object-cover" /> : <UserIcon size={14} />}</div><div><div className="font-bold text-slate-900 dark:text-white">{user.displayName}</div><div className="text-xs text-slate-500">{user.email}</div></div></div></td><td className="p-4"><button onClick={() => handleRoleChange(user.uid, user.role)} className={`px-2 py-1 rounded text-xs font-bold uppercase ${user.role === 'admin' ? 'bg-purple-100 text-purple-700 dark:bg-purple-900/30 dark:text-purple-400' : 'bg-slate-100 text-slate-600 dark:bg-slate-800 dark:text-slate-400'} `}>{user.role}</button></td><td className="p-4"><span className={`px-2 py-1 rounded text-xs font-bold ${user.isBanned ? 'bg-red-100 text-red-600 dark:bg-red-900/30 dark:text-red-400' : 'bg-green-100 text-green-600 dark:bg-green-900/30 dark:text-green-400'} `}>{user.isBanned ? 'Banned' : 'Active'}</span></td><td className="p-4 text-xs"><div className="font-bold text-slate-700 dark:text-slate-300">{((user as any).ipAddress === '::1' || (user as any).ipAddress === '::ffff:127.0.0.1') ? '127.0.0.1' : ((user as any).ipAddress || 'Unknown')}</div><div className="text-slate-500">{user.country || 'Unknown'}</div></td><td className="p-4 text-slate-500">{new Date(user.createdAt).toLocaleDateString()}</td><td className="p-4 text-right flex justify-end gap-2">
                {viewMode === 'active' ? (
                    <>
                        <button onClick={() => handleRoleChange(user.uid, user.role)} className={`p-2 rounded-lg ${user.role === 'admin' ? 'bg-purple-100 text-purple-700 dark:bg-purple-900/30 dark:text-purple-400' : 'bg-slate-100 text-slate-600 dark:bg-slate-800 dark:text-slate-400'} hover:opacity-80 transition`} title={user.role === 'admin' ? "Demote to User" : "Promote to Admin"}>{user.role === 'admin' ? <ShieldOff size={16} /> : <ShieldCheck size={16} />}</button>
                        <button onClick={() => handleMessageClick(user.uid)} className="p-2 bg-blue-50 dark:bg-blue-900/20 text-blue-600 dark:text-blue-400 rounded-lg hover:bg-blue-100 dark:hover:bg-blue-900/40" title="Send Message"><MessageSquare size={16} /></button>
                        <button onClick={() => setSelectedUser(user)} className="p-2 bg-slate-100 dark:bg-slate-800 rounded-lg hover:bg-slate-200 dark:hover:bg-slate-700 text-slate-600 dark:text-slate-400"><Eye size={16} /></button>
                        <button onClick={() => handleToggleBan(user.uid)} className={`p-2 rounded-lg ${user.isBanned ? 'bg-green-100 text-green-600 dark:bg-green-900/30 dark:text-green-400' : 'bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400'} `} title={user.isBanned ? "Unban User (Restore Access)" : "Ban User & IP"}>{user.isBanned ? <CheckCircle2 size={16} /> : <ShieldAlert size={16} />}</button>
                        <DoubleConfirmDeleteButton onDelete={() => handleDeleteUser(user.uid, user.role)} className="p-2 bg-red-50 dark:bg-red-900/20 text-red-600 dark:text-red-400 rounded-lg hover:bg-red-100 dark:hover:bg-red-900/40" />
                    </>
                ) : (
                    <div className="flex gap-2 justify-end">
                        <button onClick={() => handleRestoreUser(user.uid)} className="px-3 py-1.5 bg-emerald-100 dark:bg-emerald-900/30 text-emerald-600 dark:text-emerald-400 text-xs font-bold rounded-lg hover:bg-emerald-200 dark:hover:bg-emerald-800 transition">Restore</button>
                        <button onClick={() => handlePermanentDelete(user.uid)} className="px-3 py-1.5 bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 text-xs font-bold rounded-lg hover:bg-red-200 dark:hover:bg-red-800 transition flex items-center gap-1"><Trash2 size={12} /> Forever</button>
                    </div>
                )}</td></tr>))}</tbody></table></div>
            {selectedUser && <UserProfileModal user={selectedUser} onClose={() => setSelectedUser(null)} />}
            {notification && (
                <div className={`fixed top-24 right-4 z-[100] px-6 py-4 rounded-xl shadow-2xl flex items-center gap-3 animate-fade-in-up ${notification.type === 'error' ? 'bg-red-500 text-white' :
                    notification.type === 'warning' ? 'bg-amber-500 text-white' :
                        'bg-green-500 text-white'
                    }`}>
                    {notification.type === 'error' ? <AlertCircle size={24} /> :
                        notification.type === 'warning' ? <ShieldAlert size={24} /> :
                            <CheckCircle2 size={24} />}
                    <div>
                        <h4 className="font-bold text-lg">{notification.type === 'error' ? 'Error' : notification.type === 'warning' ? 'Security Warning' : 'Success'}</h4>
                        <p className="text-sm font-medium opacity-90">{notification.message}</p>
                    </div>
                    <button onClick={() => setNotification(null)} className="ml-4 p-1 hover:bg-white/20 rounded-full transition"><X size={18} /></button>
                </div>
            )}
        </div>
    );
};

const AnalyticsView = () => {
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [trafficData, setTrafficData] = useState<any[]>([]);
    const [modelUsageData, setModelUsageData] = useState<any[]>([]);
    const [revenueData, setRevenueData] = useState<any[]>([]);
    const [geoData, setGeoData] = useState<any[]>([]);
    const [systemLoadData, setSystemLoadData] = useState<any[]>([]);

    const fetchAnalytics = async () => {
        try {
            setError(null);
            const data = await adminService.getAnalytics();

            setTrafficData(data.weeklyTraffic || []);
            setModelUsageData(data.modelUsageDistribution || []);
            setRevenueData(data.monthlyRevenue || []);
            setGeoData(data.topCountries || []);
            setSystemLoadData(data.systemLoadData || []);

            setLoading(false);
        } catch (err: any) {
            console.error('Failed to fetch analytics:', err);
            setError(err.message || 'Failed to load analytics');
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchAnalytics();

        // Auto-refresh every 30 seconds
        const interval = setInterval(fetchAnalytics, 30000);
        return () => clearInterval(interval);
    }, []);

    if (loading) {
        return (
            <div className="flex items-center justify-center h-96">
                <Loader2 className="animate-spin text-blue-500" size={48} />
            </div>
        );
    }

    if (error) {
        return (
            <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-2xl p-8 text-center">
                <AlertCircle className="mx-auto mb-4 text-red-500" size={48} />
                <h3 className="text-lg font-bold text-red-900 dark:text-red-200 mb-2">Failed to Load Analytics</h3>
                <p className="text-red-700 dark:text-red-300 text-sm mb-4">{error}</p>
                <button
                    onClick={fetchAnalytics}
                    className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition"
                >
                    Retry
                </button>
            </div>
        );
    }

    return (
        <div className="space-y-8">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2">
                    <BarChart3 className="text-blue-500" /> Analytics Dashboard
                </h2>
                <button
                    onClick={fetchAnalytics}
                    className="p-2 hover:bg-slate-100 dark:hover:bg-slate-800 rounded-lg transition"
                    title="Refresh Analytics"
                >
                    <RefreshCw size={18} className="text-slate-600 dark:text-slate-400" />
                </button>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                {/* Weekly Traffic */}
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm h-80">
                    <h3 className="font-bold text-slate-900 dark:text-white mb-4">Weekly Traffic</h3>
                    {trafficData.length > 0 ? (
                        <ResponsiveContainer width="100%" height="100%">
                            <BarChart data={trafficData}>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.3} vertical={false} />
                                <XAxis dataKey="name" stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                <YAxis stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                <Tooltip cursor={{ fill: 'transparent' }} contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc' }} />
                                <Bar dataKey="users" fill="#3b82f6" radius={[4, 4, 0, 0]} name="Users" />
                                <Bar dataKey="sessions" fill="#8b5cf6" radius={[4, 4, 0, 0]} name="Sessions" />
                                <Legend />
                            </BarChart>
                        </ResponsiveContainer>
                    ) : (
                        <div className="flex items-center justify-center h-full text-slate-500">No traffic data</div>
                    )}
                </div>

                {/* Model Usage Distribution */}
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm h-80">
                    <h3 className="font-bold text-slate-900 dark:text-white mb-4">Model Usage Distribution</h3>
                    {modelUsageData.length > 0 ? (
                        <ResponsiveContainer width="100%" height="100%">
                            <PieChart>
                                <Pie data={modelUsageData} cx="50%" cy="50%" innerRadius={60} outerRadius={80} paddingAngle={5} dataKey="value">
                                    {modelUsageData.map((entry, index) => (
                                        <Cell key={`cell-${index}`} fill={entry.color} />
                                    ))}
                                </Pie>
                                <Tooltip contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc' }} />
                                <Legend />
                            </PieChart>
                        </ResponsiveContainer>
                    ) : (
                        <div className="flex items-center justify-center h-full text-slate-500">No model usage data</div>
                    )}
                </div>

                {/* Monthly Revenue */}
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm h-80">
                    <h3 className="font-bold text-slate-900 dark:text-white mb-4">Monthly Revenue</h3>
                    {revenueData.length > 0 ? (
                        <ResponsiveContainer width="100%" height="100%">
                            <LineChart data={revenueData}>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" strokeOpacity={0.3} vertical={false} />
                                <XAxis dataKey="name" stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                <YAxis stroke="#64748b" fontSize={12} tickLine={false} axisLine={false} />
                                <Tooltip contentStyle={{ backgroundColor: '#0f172a', borderColor: '#1e293b', borderRadius: '12px', color: '#f8fafc' }} />
                                <Line type="monotone" dataKey="revenue" stroke="#f59e0b" strokeWidth={3} dot={{ r: 4 }} />
                            </LineChart>
                        </ResponsiveContainer>
                    ) : (
                        <div className="flex items-center justify-center h-full text-slate-500">No revenue data</div>
                    )}
                </div>

                {/* Top Countries */}
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm h-80 overflow-y-auto custom-scrollbar">
                    <h3 className="font-bold text-slate-900 dark:text-white mb-4">Top Countries</h3>
                    {geoData.length > 0 ? (
                        <table className="w-full text-sm text-left">
                            <thead className="text-slate-500 border-b border-slate-200 dark:border-slate-800">
                                <tr>
                                    <th className="pb-2">Country</th>
                                    <th className="pb-2">Users</th>
                                    <th className="pb-2 text-right">Trend</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-slate-100 dark:divide-slate-800">
                                {geoData.map((geo, i) => (
                                    <tr key={i}>
                                        <td className="py-3 font-medium text-slate-900 dark:text-slate-200">{geo.country}</td>
                                        <td className="py-3 text-slate-600 dark:text-slate-400">{geo.users}</td>
                                        <td className={`py-3 text-right font-bold ${geo.trend.startsWith('+') ? 'text-green-500' : 'text-red-500'}`}>{geo.trend}</td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    ) : (
                        <div className="flex items-center justify-center h-full text-slate-500">No geographic data</div>
                    )}
                </div>
            </div>
        </div>
    );
};

const MessagesView = ({ selectedUserId, setSelectedUserId }: { selectedUserId: string | null, setSelectedUserId: (id: string | null) => void }) => {
    console.log("MessagesView rendered with selectedUserId:", selectedUserId);
    const [messages, setMessages] = useState<SupportMessage[]>([]);
    const [users, setUsers] = useState<User[]>([]);
    const [input, setInput] = useState('');
    const scrollRef = useRef<HTMLDivElement>(null);

    useEffect(() => {
        const fetch = async () => {
            const [msgs, userList] = await Promise.all([
                adminService.getSupportMessages(),
                adminService.getUsers()
            ]);
            setMessages(msgs);
            setUsers(userList);
        };
        fetch();
        const interval = setInterval(async () => {
            const msgs = await adminService.getSupportMessages();
            setMessages(msgs);
        }, 5000);
        return () => clearInterval(interval);
    }, []);

    useEffect(() => {
        scrollRef.current?.scrollIntoView({ behavior: 'smooth' });

        // Mark messages as read when viewing a thread
        const markMessagesRead = async () => {
            if (!selectedUserId) return;

            const unreadUserMessages = messages.filter(m =>
                m.userId === selectedUserId &&
                m.sender === 'user' &&
                m.status !== 'read'
            );

            if (unreadUserMessages.length === 0) return;

            // Optimistically update local state
            setMessages(prev => prev.map(m =>
                m.userId === selectedUserId && m.sender === 'user' && m.status !== 'read'
                    ? { ...m, status: 'read' as const }
                    : m
            ));

            // Optimistically update App.tsx state
            window.dispatchEvent(new CustomEvent('sf_admin_messages_read', { detail: unreadUserMessages.map(m => m.id) }));

            // Mark in backend
            const markReadPromises = unreadUserMessages.map(msg =>
                adminService.markMessageAsRead(msg.id)
                    .catch(error => console.error('Failed to mark message as read:', error))
            );

            await Promise.all(markReadPromises);
            window.dispatchEvent(new Event('sf_notifications_updated'));
        };

        markMessagesRead();
    }, [selectedUserId, messages]);

    const handleSend = async () => {
        if (!selectedUserId || !input.trim()) return;

        const targetUser = users.find(u => u.uid === selectedUserId);
        if (!targetUser) return;

        const newMessage: SupportMessage = {
            id: Date.now().toString(),
            sender: 'admin',
            text: input,
            timestamp: Date.now(),
            status: 'sent',
            userId: targetUser.uid,
            userEmail: targetUser.email,
            userName: targetUser.displayName
        };

        await adminService.sendSupportMessage(newMessage);
        setMessages(prev => [...prev, newMessage]);
        setInput('');
    };

    const handleDeleteConversation = async (userId: string) => {
        await adminService.deleteConversation(userId);
        setMessages(prev => prev.filter(m => m.userId !== userId));
        if (selectedUserId === userId) setSelectedUserId(null);
    };

    const handleDeleteMessage = async (msgId: string) => {
        await adminService.deleteSupportMessage(msgId);
        setMessages(prev => prev.filter(m => m.id !== msgId));
    };

    const threads = Object.values(messages.reduce((acc: any, msg) => {
        if (!acc[msg.userId]) {
            acc[msg.userId] = {
                userId: msg.userId,
                email: msg.userEmail,
                messages: [],
                lastMsg: msg
            };
        }
        acc[msg.userId].messages.push(msg);
        if (msg.timestamp > acc[msg.userId].lastMsg.timestamp) {
            acc[msg.userId].lastMsg = msg;
        }
        return acc;
    }, {}));

    const activeThreadMessages = selectedUserId
        ? messages.filter(m => m.userId === selectedUserId).sort((a, b) => a.timestamp - b.timestamp)
        : [];

    const activeUser = users.find(u => u.uid === selectedUserId);

    return (
        <div className="h-[calc(100vh-200px)] min-h-[600px] flex bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden shadow-sm">
            {/* Left: Threads List */}
            <div className="w-1/3 border-r border-slate-200 dark:border-slate-800 flex flex-col">
                <div className="p-4 border-b border-slate-200 dark:border-slate-800 bg-slate-50 dark:bg-slate-950/50">
                    <h3 className="font-bold text-slate-900 dark:text-white flex items-center gap-2">
                        <MessageSquare className="text-blue-500" size={18} /> Inbox
                    </h3>
                </div>
                <div className="flex-1 overflow-y-auto custom-scrollbar">
                    {threads.length === 0 && <div className="p-8 text-center text-slate-500 text-sm">No messages yet.</div>}
                    {threads.map((thread: any) => (
                        <div key={thread.userId} className="relative group">
                            <button
                                onClick={() => setSelectedUserId(thread.userId)}
                                className={`w-full text-left p-4 border-b border-slate-100 dark: border-slate-800/50 hover: bg-slate-50 dark: hover: bg-slate-800/50 transition-colors ${selectedUserId === thread.userId ? 'bg-blue-50 dark:bg-blue-900/10 border-l-4 border-l-blue-500' : 'border-l-4 border-l-transparent'} `}
                            >
                                <div className="flex justify-between mb-1">
                                    <span className={`font-bold text-sm truncate ${selectedUserId === thread.userId ? 'text-blue-700 dark:text-blue-400' : 'text-slate-900 dark:text-slate-200'} `}>
                                        {thread.email}
                                    </span>
                                    <span className="text-[10px] text-slate-400 whitespace-nowrap">
                                        {new Date(thread.lastMsg.timestamp).toLocaleDateString()}
                                    </span>
                                </div>
                                <p className="text-xs text-slate-500 dark:text-slate-400 line-clamp-1">{thread.lastMsg.sender === 'admin' ? 'You: ' : ''}{thread.lastMsg.text}</p>
                            </button>
                            <div className="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity z-10">
                                <DoubleConfirmDeleteButton
                                    onDelete={() => handleDeleteConversation(thread.userId)}
                                    className="p-1.5 bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 rounded-lg"
                                    iconSize={14}
                                />
                            </div>
                        </div>
                    ))}
                </div>
                {/* User Selector for New Message */}
                <div className="p-4 border-t border-slate-200 dark:border-slate-800 bg-slate-50 dark:bg-slate-950/50">
                    <div className="relative">
                        <Search className="absolute left-3 top-2.5 text-slate-400" size={14} />
                        <select
                            className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl pl-9 pr-4 py-2 text-xs outline-none focus:border-blue-500 text-slate-700 dark:text-slate-300 appearance-none cursor-pointer"
                            onChange={(e) => { if (e.target.value) setSelectedUserId(e.target.value); }}
                            value=""
                        >
                            <option value="" disabled>Message a user...</option>
                            {users.map(u => (
                                <option key={u.uid} value={u.uid}>{u.displayName} ({u.email})</option>
                            ))}
                        </select>
                    </div>
                </div>
            </div>

            {/* Right: Chat Area */}
            <div className="flex-1 flex flex-col bg-slate-50/30 dark:bg-slate-950/30">
                {selectedUserId ? (
                    <>
                        <div className="p-4 border-b border-slate-200 dark:border-slate-800 bg-white dark:bg-slate-900 flex justify-between items-center">
                            <div>
                                <h4 className="font-bold text-slate-900 dark:text-white">{activeUser?.displayName || 'Unknown User'}</h4>
                                <p className="text-xs text-slate-500">{activeUser?.email}</p>
                            </div>
                            <span className={`px-2 py-1 rounded text - [10px] font-bold uppercase ${activeUser?.role === 'admin' ? 'bg-purple-100 text-purple-700' : 'bg-slate-100 text-slate-600'} `}>{activeUser?.role || 'User'}</span>
                        </div>

                        <div className="flex-1 overflow-y-auto p-4 space-y-4 custom-scrollbar">
                            {activeThreadMessages.length === 0 && (
                                <div className="p-8 text-center text-slate-500 text-sm">
                                    <MessageSquare className="mx-auto mb-4 opacity-20" size={48} />
                                    <p>No messages yet. Send a message to start the conversation.</p>
                                </div>
                            )}
                            {activeThreadMessages.map(msg => (
                                <div key={msg.id} className={`group flex flex-col ${msg.sender === 'admin' ? 'items-end' : 'items-start'} relative`}>
                                    <span className={`text - [10px] font-bold uppercase tracking-wider mb-1 px-2 ${msg.sender === 'admin' ? 'text-blue-600 dark:text-blue-400' : 'text-slate-600 dark:text-slate-400'} `}>
                                        {msg.sender === 'admin' ? 'You (Admin)' : (msg.userName || msg.userEmail || 'User')}
                                    </span>
                                    <div className={`max-w - [75 %] p-3 rounded-2xl text-sm ${msg.sender === 'admin'
                                        ? 'bg-blue-600 text-white rounded-br-none'
                                        : 'bg-slate-100 dark:bg-slate-800 text-slate-900 dark:text-slate-200 border border-slate-200 dark:border-slate-700 rounded-bl-none'
                                        } `}>
                                        <p>{msg.text}</p>
                                        <p className={`text - [10px] mt-1 opacity-70 ${msg.sender === 'admin' ? 'text-blue-100' : 'text-slate-400'} `}>
                                            {new Date(msg.timestamp).toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                                        </p>
                                    </div>
                                    <button
                                        onClick={() => handleDeleteMessage(msg.id)}
                                        className="absolute top-0 right-0 opacity-0 group-hover:opacity-100 p-1 bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 rounded-lg transition-opacity"
                                        title="Delete message"
                                    >
                                        <X size={12} />
                                    </button>
                                </div>
                            ))}
                            <div ref={scrollRef}></div>
                        </div>

                        <div className="p-4 bg-white dark:bg-slate-900 border-t border-slate-200 dark:border-slate-800">
                            <div className="flex gap-2">
                                <input
                                    type="text"
                                    value={input}
                                    onChange={e => setInput(e.target.value)}
                                    onKeyDown={e => e.key === 'Enter' && handleSend()}
                                    placeholder="Type a reply..."
                                    className="flex-1 bg-slate-100 dark:bg-slate-800 border-none rounded-xl px-4 py-3 text-sm focus:ring-2 focus:ring-blue-500 outline-none text-slate-900 dark:text-white"
                                />
                                <button onClick={handleSend} disabled={!input.trim()} className="p-3 bg-blue-600 hover:bg-blue-500 text-white rounded-xl transition disabled:opacity-50 shadow-lg shadow-blue-500/20">
                                    <Send size={18} />
                                </button>
                            </div>
                        </div>
                    </>
                ) : (
                    <div className="flex-1 flex flex-col items-center justify-center text-slate-400">
                        <MessageSquare size={48} className="mb-4 opacity-20" />
                        <p>Select a conversation or start a new one.</p>
                    </div>
                )}
            </div>
        </div>
    );
};

const BroadcastsView = () => {
    const [broadcasts, setBroadcasts] = useState<Broadcast[]>([]);
    const [title, setTitle] = useState('');
    const [content, setContent] = useState('');
    const [type, setType] = useState<'announcement' | 'feature' | 'notice' | 'urgent'>('announcement');
    const [targetAudience, setTargetAudience] = useState<string>('All');
    const [isSending, setIsSending] = useState(false);


    const loadBroadcasts = async () => {
        const list = await adminService.getBroadcasts();
        // Filter out deleted ones for display, but maybe keep them for stats? 
        // Usually stats include history. But let's stick to visible history.
        const activeList = list.filter(b => !b.isDeleted).sort((a, b) => b.sentAt - a.sentAt);
        setBroadcasts(activeList);


    };

    useEffect(() => {
        loadBroadcasts();
    }, []);

    const handleSend = async () => {
        if (!title || !content) return;
        setIsSending(true);

        const newBroadcast: Broadcast = {
            id: '', // Empty ID tells service to create new
            title,
            content,
            type,
            targetAudience,
            sentAt: Date.now(),
            recipients: 0, // Calculated by server usually
            openRate: 0,
            readCount: 0,
            isDeleted: false
        };

        try {
            await adminService.saveBroadcast(newBroadcast);
            setTitle('');
            setContent('');
            setType('announcement');
            setTargetAudience('All');
            await loadBroadcasts();
            // alert("Broadcast sent successfully!"); // Removed per user request
        } catch (error) {
            console.error("Failed to send broadcast", error);
            alert("Failed to send broadcast.");
        } finally {
            setIsSending(false);
        }
    };

    const handleDelete = async (id: string) => {
        try {
            await adminService.deleteBroadcast(id);
            await loadBroadcasts();
        } catch (error) {
            console.error("Delete failed", error);
            alert("Failed to delete broadcast");
        }
    };

    // Stats
    const totalSent = broadcasts.length;
    const totalReach = broadcasts.reduce((acc, b) => acc + (b.recipients || 0), 0);
    const avgOpenRate = totalSent > 0 ? Math.round(broadcasts.reduce((acc, b) => acc + (b.openRate || 0), 0) / totalSent) : 0;

    const [isRefreshing, setIsRefreshing] = useState(false);

    const handleRefresh = async () => {
        setIsRefreshing(true);
        await loadBroadcasts();
        setTimeout(() => setIsRefreshing(false), 500); // Small delay for visual feedback
    };

    return (
        <div className="space-y-8 relative">
            {/* Header Stats */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 p-6 rounded-2xl shadow-sm flex items-center gap-4">
                    <div className="p-3 bg-blue-100 dark:bg-blue-900/30 text-blue-600 dark:text-blue-400 rounded-xl">
                        <Megaphone size={24} />
                    </div>
                    <div>
                        <p className="text-xs font-bold text-slate-500 uppercase">Total Broadcasts</p>
                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white">{totalSent}</h3>
                    </div>
                </div>
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 p-6 rounded-2xl shadow-sm flex items-center gap-4">
                    <div className="p-3 bg-purple-100 dark:bg-purple-900/30 text-purple-600 dark:text-purple-400 rounded-xl">
                        <Users size={24} />
                    </div>
                    <div>
                        <p className="text-xs font-bold text-slate-500 uppercase">Total Reach</p>
                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white">
                            {totalReach >= 1000 ? `${(totalReach / 1000).toFixed(1)} k` : totalReach}
                        </h3>
                    </div>
                </div>
                <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 p-6 rounded-2xl shadow-sm flex items-center gap-4">
                    <div className="p-3 bg-green-100 dark:bg-green-900/30 text-green-600 dark:text-green-400 rounded-xl">
                        <BarChart3 size={24} />
                    </div>
                    <div>
                        <p className="text-xs font-bold text-slate-500 uppercase">Avg. Open Rate</p>
                        <h3 className="text-2xl font-bold text-slate-900 dark:text-white">{avgOpenRate}%</h3>
                    </div>
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                {/* Compose Area */}
                <div className="lg:col-span-1 space-y-6">
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm">
                        <h3 className="font-bold text-slate-900 dark:text-white mb-6 flex items-center gap-2">
                            <Send className="text-amber-500" size={24} /> Compose Broadcast
                        </h3>
                        <div className="space-y-4">
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Title</label>
                                <input
                                    type="text"
                                    value={title} onChange={e => setTitle(e.target.value)}
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-sm outline-none focus:ring-2 focus:ring-amber-500/20 focus:border-amber-500 transition-all font-bold text-slate-900 dark:text-white placeholder-slate-400"
                                    placeholder="e.g. New Feature Alert: AI Upscaling"
                                />
                            </div>
                            <div>
                                <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Message</label>
                                <textarea
                                    value={content} onChange={e => setContent(e.target.value)}
                                    rows={4}
                                    className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-sm outline-none focus:ring-2 focus:ring-amber-500/20 focus:border-amber-500 transition-all font-medium text-slate-900 dark:text-white placeholder-slate-400 resize-none"
                                    placeholder="Write your announcement here..."
                                />
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Type</label>
                                    <div className="relative">
                                        <select
                                            value={type} onChange={e => setType(e.target.value as any)}
                                            className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-sm outline-none cursor-pointer appearance-none"
                                        >
                                            <option value="announcement">Announcement</option>
                                            <option value="feature">New Feature</option>
                                            <option value="notice">Notice</option>
                                            <option value="urgent">Urgent</option>
                                        </select>
                                        <div className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none text-slate-500">
                                            <ChevronDown size={16} />
                                        </div>
                                    </div>
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Target Audience</label>
                                    <div className="relative">
                                        <select
                                            value={targetAudience} onChange={e => setTargetAudience(e.target.value)}
                                            className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-xl p-3 text-sm outline-none cursor-pointer appearance-none"
                                        >
                                            <option value="All">All Users</option>
                                            <option value="Free">Free Plan</option>
                                            <option value="Pro">Pro Plan</option>
                                            <option value="Enterprise">Enterprise</option>
                                        </select>
                                        <div className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none text-slate-500">
                                            <ChevronDown size={16} />
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <button
                                onClick={handleSend}
                                disabled={isSending || !title || !content}
                                className="w-full py-3 bg-gradient-to-r from-amber-500 to-orange-600 hover:from-amber-600 hover:to-orange-700 text-white font-bold rounded-xl transition flex items-center justify-center gap-2 disabled:opacity-50 shadow-lg shadow-amber-500/20 mt-2"
                            >
                                {isSending ? <Loader2 className="animate-spin" /> : <Send size={18} />} Send Broadcast
                            </button>
                        </div>
                    </div>

                    {/* Preview Card */}
                    <div className="bg-slate-100 dark:bg-slate-950/50 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 border-dashed">
                        <h4 className="text-xs font-bold text-slate-400 uppercase mb-4 text-center">Live Preview</h4>
                        <div className="bg-white dark:bg-slate-900 rounded-xl p-4 shadow-sm border border-slate-200 dark:border-slate-800 relative overflow-hidden">
                            <div className={`absolute left-0 top-0 bottom-0 w-1 ${type === 'urgent' ? 'bg-red-500' : type === 'feature' ? 'bg-blue-500' : 'bg-amber-500'} `}></div>
                            <h4 className="font-bold text-slate-900 dark:text-white text-lg mb-1">{title || 'Broadcast Title'}</h4>
                            <p className="text-sm text-slate-600 dark:text-slate-300">{content || 'Your message content will appear here...'}</p>
                        </div>
                    </div>
                </div>

                {/* History Timeline */}
                <div className="lg:col-span-2">
                    <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-6 shadow-sm min-h-[600px] relative">
                        <h3 className="font-bold text-slate-900 dark:text-white text-xl mb-6 flex items-center gap-2">
                            <History className="text-slate-400" size={24} /> Broadcast History
                        </h3>
                        <div className="space-y-6 relative before:absolute before:left-4 before:top-2 before:bottom-2 before:w-0.5 before:bg-slate-100 dark:before:bg-slate-800 pb-12">
                            {broadcasts.length === 0 && <div className="text-center text-slate-500 p-10 ml-8">No broadcasts sent yet.</div>}
                            {broadcasts.map(b => (
                                <div key={b.id} className="relative pl-12 group">
                                    <div className={`absolute left - [11px] top-4 w-3 h-3 rounded-full border-2 border-white dark: border-slate-900 ${b.type === 'urgent' ? 'bg-red-500' : b.type === 'feature' ? 'bg-blue-500' : 'bg-amber-500'} `}></div>
                                    <div className="bg-slate-50 dark:bg-slate-950/50 border border-slate-200 dark:border-slate-800 rounded-xl p-5 hover:bg-white dark:hover:bg-slate-900 hover:shadow-md transition-all duration-200">
                                        <div className="flex justify-between items-start mb-3">
                                            <div>
                                                <div className="flex items-center gap-2 mb-1">
                                                    <span className={`px-2 py-0.5 rounded text - [10px] font-bold uppercase ${b.type === 'urgent' ? 'bg-red-100 text-red-600 dark:bg-red-900/30 dark:text-red-400' :
                                                        b.type === 'feature' ? 'bg-blue-100 text-blue-600 dark:bg-blue-900/30 dark:text-blue-400' :
                                                            'bg-amber-100 text-amber-600 dark:bg-amber-900/30 dark:text-amber-400'
                                                        } `}>{b.type}</span>
                                                    <span className="text-xs text-slate-400">{new Date(b.sentAt).toLocaleString()}</span>
                                                </div>
                                                <h4 className="font-bold text-slate-900 dark:text-white text-lg">{b.title}</h4>
                                            </div>
                                            <DoubleConfirmDeleteButton onDelete={() => handleDelete(b.id)} className="p-2 text-slate-400 hover:text-red-500 transition rounded-lg hover:bg-slate-100 dark:hover:bg-slate-800 opacity-0 group-hover:opacity-100" />
                                        </div>
                                        <p className="text-sm text-slate-600 dark:text-slate-300 mb-4 leading-relaxed">{b.content}</p>

                                        <div className="flex items-center gap-6 pt-4 border-t border-slate-200 dark:border-slate-800">
                                            <div>
                                                <p className="text-[10px] font-bold text-slate-400 uppercase">Recipients</p>
                                                <p className="text-sm font-bold text-slate-700 dark:text-slate-200">{b.recipients?.toLocaleString() || '-'}</p>
                                            </div>
                                            <div>
                                                <p className="text-[10px] font-bold text-slate-400 uppercase">Open Rate</p>
                                                <p className="text-sm font-bold text-slate-700 dark:text-slate-200">{b.openRate ? `${b.openRate}% ` : '-'}</p>
                                            </div>
                                            <div>
                                                <p className="text-[10px] font-bold text-slate-400 uppercase">Open Click</p>
                                                <p className="text-sm font-bold text-slate-700 dark:text-slate-200">{b.readCount || 0}</p>
                                            </div>
                                            <div>
                                                <p className="text-[10px] font-bold text-slate-400 uppercase">Target</p>
                                                <p className="text-sm font-bold text-slate-700 dark:text-slate-200 capitalize">{b.targetAudience}</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </div>
                </div>
            </div>

            {/* Global Refresh Button */}
            <button
                onClick={handleRefresh}
                className={`fixed bottom-10 right-10 p-4 bg-slate-900 dark: bg-white text-white dark: text-slate-900 rounded-full shadow-2xl hover: scale-110 transition-all z-50 flex items-center justify-center ${isRefreshing ? 'animate-spin' : ''} `}
                title="Refresh Page Data"
            >
                <RefreshCw size={24} />
            </button>
        </div>
    );
};

const InboxView = () => {
    const [emails, setEmails] = useState<SystemEmail[]>([]);
    useEffect(() => { adminService.getEmailLogs().then(setEmails); }, []);

    return (
        <div className="space-y-6">
            <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><Inbox className="text-purple-500" /> System Email Log</h2>
            <div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl overflow-hidden shadow-sm"><div className="overflow-x-auto"><table className="w-full text-sm text-left whitespace-nowrap"><thead className="bg-slate-50 dark:bg-slate-950 text-slate-500 border-b border-slate-200 dark:border-slate-800"><tr><th className="p-4">To</th><th className="p-4">Subject</th><th className="p-4">Type</th><th className="p-4">Status</th><th className="p-4 text-right">Time</th></tr></thead><tbody className="divide-y divide-slate-100 dark:divide-slate-800">{emails.map(email => (<tr key={email.id} className="hover:bg-slate-50 dark:hover:bg-slate-800/50"><td className="p-4 font-medium text-slate-900 dark:text-white">{email.to}</td><td className="p-4 text-slate-600 dark:text-slate-300">{email.subject}</td><td className="p-4"><span className="px-2 py-1 rounded bg-slate-100 dark:bg-slate-800 text-xs font-bold uppercase">{email.type}</span></td><td className="p-4"><span className="text-green-500 font-bold text-xs flex items-center gap-1"><CheckCircle2 size={12} /> Sent</span></td><td className="p-4 text-right text-slate-500">{new Date(email.sentAt).toLocaleTimeString()}</td></tr>))}{emails.length === 0 && <tr><td colSpan={5} className="p-8 text-center text-slate-500">No emails sent yet.</td></tr>}</tbody></table></div></div>
        </div>
    );
};

const EmailSettingsView = () => {
    const [settings, setSettings] = useState<SiteSettings | null>(null);
    const [isSaving, setIsSaving] = useState(false);
    useEffect(() => { adminService.getSiteSettings().then(setSettings); }, []);
    const handleSave = async () => { if (settings) { setIsSaving(true); await adminService.saveSiteSettings(settings); setIsSaving(false); } };
    if (!settings) return <div>Loading...</div>;

    return (
        <div className="space-y-8 max-w-3xl"><div className="flex justify-between items-center"><h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><Mail className="text-amber-500" /> Email Configuration</h2><button onClick={handleSave} disabled={isSaving} className="px-6 py-2 bg-amber-500 hover:bg-amber-600 text-slate-950 font-bold rounded-lg transition disabled:opacity-50">{isSaving ? 'Saving...' : 'Save Changes'}</button></div><div className="bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl p-8 space-y-6 shadow-sm"><div className="grid grid-cols-2 gap-6"><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">SMTP Host</label><input type="text" value={settings.smtpConfig?.host} onChange={e => setSettings({ ...settings, smtpConfig: { ...settings.smtpConfig, host: e.target.value } })} className="w-full bg-slate-50 dark:bg-slate-950 border border-slate-200 dark:border-slate-700 rounded-lg p-3 text-sm outline-none focus:border-amber-500" /></div><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">SMTP Port</label><input type="number" value={settings.smtpConfig?.port} onChange={e => setSettings({ ...settings, smtpConfig: { ...settings.smtpConfig, port: parseInt(e.target.value) } })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-3 text-sm outline-none focus:border-amber-500" /></div></div><div className="grid grid-cols-2 gap-6"><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Username</label><input type="text" value={settings.smtpConfig?.user} onChange={e => setSettings({ ...settings, smtpConfig: { ...settings.smtpConfig, user: e.target.value } })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-3 text-sm outline-none focus:border-amber-500" /></div><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">Password</label><input type="password" value={settings.smtpConfig?.pass} onChange={e => setSettings({ ...settings, smtpConfig: { ...settings.smtpConfig, pass: e.target.value } })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-3 text-sm outline-none focus:border-amber-500" /></div></div><div className="grid grid-cols-2 gap-6"><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">From Name</label><input type="text" value={settings.smtpConfig?.fromName} onChange={e => setSettings({ ...settings, smtpConfig: { ...settings.smtpConfig, fromName: e.target.value } })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-3 text-sm outline-none focus:border-amber-500" /></div><div><label className="block text-xs font-bold text-slate-500 uppercase mb-1.5">From Email</label><input type="email" value={settings.smtpConfig?.fromEmail} onChange={e => setSettings({ ...settings, smtpConfig: { ...settings.smtpConfig, fromEmail: e.target.value } })} className="w-full bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-700 rounded-lg p-3 text-sm outline-none focus:border-amber-500" /></div></div></div></div>
    );
};

const AuditLogView = () => {
    const [logs, setLogs] = useState<AuditLog[]>([]);
    const [isLoading, setIsLoading] = useState(false);

    const fetchLogs = async () => {
        setIsLoading(true);
        try {
            const data = await adminService.getAuditLogs();
            setLogs(data);
        } catch (error) {
            console.error("Failed to fetch audit logs:", error);
        } finally {
            setIsLoading(false);
        }
    };

    useEffect(() => { fetchLogs(); }, []);

    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <h2 className="text-2xl font-bold text-slate-900 dark:text-white flex items-center gap-2"><FileText className="text-slate-500" /> System Audit Log</h2>
                <button onClick={fetchLogs} className="p-2 text-slate-500 hover:text-slate-900 dark:hover:text-white transition rounded-lg hover:bg-slate-100 dark:hover:bg-slate-800">
                    <RefreshCw size={18} className={isLoading ? 'animate-spin' : ''} />
                </button>
            </div>
            <div className="bg-slate-900 text-slate-300 font-mono text-xs p-6 rounded-2xl border border-slate-800 overflow-y-auto max-h-[600px] shadow-inner custom-scrollbar">
                {logs.map(log => (
                    <div key={log.id} className="mb-2 border-b border-slate-800/50 pb-2 last:border-0 hover:bg-slate-800/30 transition-colors p-1 rounded">
                        <span className="text-slate-500 mr-3">[{new Date(log.timestamp).toLocaleString()}]</span>
                        <span className="text-blue-400 font-bold mr-3 uppercase tracking-wider">{log.action}</span>
                        <span className="text-amber-500 mr-3">@{log.admin}</span>
                        <span className="text-slate-600 mr-3">→</span>
                        <span className="text-white">{log.details}</span>
                        <span className="float-right text-slate-600 text-[10px] mt-0.5">IP: {log.ip}</span>
                    </div>
                ))}
                {logs.length === 0 && !isLoading && <div className="text-slate-600 italic p-4 text-center">// No audit logs found. Actions will appear here.</div>}
                {isLoading && logs.length === 0 && <div className="text-slate-500 italic p-4 text-center">Loading system logs...</div>}
            </div>
        </div>
    );
};

export default AdminDashboard;
