
import React, { useState, useEffect, useRef } from 'react';
import { AppView, AuthView, User, AdminSection, SiteSettings } from './types';
import { authService } from './services/authService';
import { adminService } from './services/adminService';
import Sidebar from './components/Sidebar';
import TopBar from './components/TopBar';
import Dashboard from './components/Dashboard';
import MarketIntelligence from './components/MarketIntelligence';
import ImageGenerator from './components/ImageGenerator';
import VideoGenerator from './components/VideoGenerator';
import MetadataGenerator from './components/MetadataGenerator';
import ImageEditor from './components/ImageEditor';
import SettingsPage from './components/SettingsPage';
import AdminDashboard from './components/AdminDashboard';
import LandingPage from './components/LandingPage';
import AuthPage from './components/auth/AuthPage';
import Library from './components/Library';
import StockPhotoReviewer from './components/StockPhotoReviewer';
import PromptGenerator from './components/PromptGenerator';
import { Folder, Wrench, ArrowUp, ScanEye } from 'lucide-react';
import { BatchProvider } from './contexts/BatchContext';

function App() {
  const [appView, setAppView] = useState<AppView>(AppView.LANDING);
  const [currentView, setCurrentView] = useState<AppView>(() => {
    const saved = localStorage.getItem('sf_current_view');
    return (saved as AppView) || AppView.DASHBOARD;
  });
  const [authView, setAuthView] = useState<AuthView>(AuthView.LOGIN);
  const [user, setUser] = useState<User | null>(null);
  const [authLoading, setAuthLoading] = useState(true); // Add loading state


  // Settings Tab State
  const [settingsActiveTab, setSettingsActiveTab] = useState<string>(() => {
    return localStorage.getItem('sf_settings_active_tab') || 'profile';
  });

  // Admin State
  const [adminSection, setAdminSection] = useState<AdminSection>(AdminSection.OVERVIEW);
  const [isViewingAsUser, setIsViewingAsUser] = useState(false); // State for "View As" mode

  // Sidebar State
  const [isSidebarCollapsed, setIsSidebarCollapsed] = useState(false);
  const [isMobileSidebarOpen, setIsMobileSidebarOpen] = useState(false);

  // Scroll Top State
  const [showScrollTop, setShowScrollTop] = useState(false);
  const mainContentRef = useRef<HTMLDivElement>(null);

  // Global App Config (fetched from admin settings)
  const [appConfig, setAppConfig] = useState<any>({});
  const [siteSettings, setSiteSettings] = useState<SiteSettings | null>(null);

  // Unread Messages Count (for notifications)
  const [unreadMessagesCount, setUnreadMessagesCount] = useState(0);
  const recentlyReadRef = useRef<Set<string>>(new Set());

  // Persistence Effects
  useEffect(() => {
    localStorage.setItem('sf_current_view', currentView);
  }, [currentView]);

  useEffect(() => {
    localStorage.setItem('sf_settings_active_tab', settingsActiveTab);
  }, [settingsActiveTab]);

  useEffect(() => {
    // Set up Firebase Auth State Observer
    const unsubscribe = authService.onAuthStateChanged(async (firebaseUser) => {
      if (firebaseUser) {
        // User is logged in
        try {
          // Use silent=true to prevent double dispatch (onAuthStateChanged already dispatches)
          const fullProfile = await authService.getUserProfile(firebaseUser.uid, true);

          // Check for Admin View As Override
          const urlParams = new URLSearchParams(window.location.search);
          const viewAsUid = urlParams.get('admin_view_as');
          let effectiveUser = fullProfile;
          let viewingAs = false;

          if (fullProfile?.role === 'admin' && viewAsUid) {
            try {
              const targetUser = await authService.getUserProfile(viewAsUid);
              if (targetUser) {
                effectiveUser = targetUser;
                viewingAs = true;
                // Force dashboard view when impersonating
                setAppView(AppView.DASHBOARD);
                setCurrentView(AppView.DASHBOARD);
              }
            } catch (e) {
              console.error("View As failed", e);
            }
          }

          if (effectiveUser) {
            setUser(effectiveUser);
            setIsViewingAsUser(viewingAs);

            if (!viewingAs) {
              // strict normal flow
              setAppView(AppView.DASHBOARD);
            }

            // NOTE: checkDailyReset is called separately in useEffect below to avoid infinite loop
            // Use actual firebase UID for location update to avoid messing with target user's location?
            // Actually, we probably don't want to update location if viewing as someone else.
            if (!viewingAs) {
              authService.updateUserLocation(firebaseUser.uid);
            }

            // Apply User Preferences (Theme & Reduced Motion)
            authService.getUserPreferences(effectiveUser.uid).then((prefs: any) => {
              if (prefs?.uiConfig?.reducedMotion) document.documentElement.classList.add('motion-reduce');
              else document.documentElement.classList.remove('motion-reduce');

              if (prefs?.theme) {
                if (prefs.theme === 'dark') {
                  document.documentElement.classList.add('dark');
                  document.documentElement.classList.remove('light');
                } else {
                  document.documentElement.classList.remove('dark');
                  document.documentElement.classList.add('light');
                }
              }
            }).catch(err => console.warn('Failed to load prefs', err));
          } else {
            // Fallback if firestore doc missing (rare)
            const currentUser = authService.getCurrentUser();
            if (currentUser) {
              setUser(currentUser);
              setAppView(AppView.DASHBOARD);
            }
          }
        } catch (e) {
          console.error("Failed to load user profile:", e);
          // Fallback to auth user if firestore fails
          const currentUser = authService.getCurrentUser();
          if (currentUser) {
            setUser(currentUser);
            setAppView(AppView.DASHBOARD);
          }
        }
        setAuthLoading(false); // Auth check complete
      } else {
        // User is logged out
        setUser(null);
        document.documentElement.classList.remove('motion-reduce');
        if (appView !== AppView.LANDING) {
          setAppView(AppView.LANDING);
        }
        setAuthLoading(false); // Auth check complete
      }
    });



    // Fetch app global config
    adminService.getLandingContent().then(content => {
      setAppConfig(content.general || {});
    }).catch(err => console.warn('[App] Failed to load landing content'));
    adminService.getSiteSettings().then(settings => {
      setSiteSettings(settings);
    }).catch(err => console.warn('[App] Failed to load site settings'));

    // Listen for user profile updates from settings page
    const handleUserUpdate = (e: Event) => {
      const customEvent = e as CustomEvent;
      const updatedUser = customEvent.detail;
      if (updatedUser) {
        setUser(updatedUser);
      } else {
        // User deleted or logged out via service
        setUser(null);
        setAuthView(AuthView.LOGIN);
        setAppView(AppView.AUTH);
      }
    };

    // Listen for navigation requests
    const handleNavigate = (e: CustomEvent) => {
      if (e.detail === 'settings_api') {
        setSettingsActiveTab('api');
        setCurrentView(AppView.SETTINGS);
      } else if (e.detail && e.detail.startsWith('settings_')) {
        // Generic settings navigation
        setSettingsActiveTab(e.detail.replace('settings_', ''));
        setCurrentView(AppView.SETTINGS);
      }
    };

    // Listen for admin section navigation
    const handleAdminNavigate = (e: CustomEvent) => {
      console.log("App.tsx received sf_navigate_admin:", e.detail);
      if (e.detail && setAdminSection) {
        // Navigate to admin view and specific section
        setCurrentView(AppView.ADMIN);
        setAdminSection(e.detail as AdminSection);
      }
    };

    window.addEventListener('sf_user_updated', handleUserUpdate);
    window.addEventListener('sf_navigate', handleNavigate as EventListener);
    window.addEventListener('sf_navigate_admin', handleAdminNavigate as EventListener);

    // Listen for settings updates
    const handleSettingsUpdate = (e: Event) => {
      const customEvent = e as CustomEvent;
      if (customEvent.detail) {
        setSiteSettings(customEvent.detail);
      }
    };
    window.addEventListener('sf_settings_updated', handleSettingsUpdate);

    // Poll for Daily Reset every 5 minutes (to handle midnight rollover)
    const resetInterval = setInterval(() => {
      if (user) {
        authService.checkDailyReset(user.uid);
      }
    }, 5 * 60 * 1000);

    return () => {
      unsubscribe(); // Unsubscribe from auth state observer
      clearInterval(resetInterval);
      window.removeEventListener('sf_user_updated', handleUserUpdate);
      window.removeEventListener('sf_navigate', handleNavigate as EventListener);
      window.removeEventListener('sf_navigate_admin', handleAdminNavigate as EventListener);
      window.removeEventListener('sf_settings_updated', handleSettingsUpdate);
    }
  }, []);

  // Update Favicon dynamically
  useEffect(() => {
    if (siteSettings?.appIcon) {
      const link = document.querySelector("link[rel*='icon']") as HTMLLinkElement || document.createElement('link');
      link.type = 'image/x-icon';
      link.rel = 'icon';
      link.href = siteSettings.appIcon;
      document.getElementsByTagName('head')[0].appendChild(link);
    }
  }, [siteSettings?.appIcon]);



  // Poll for unread messages
  useEffect(() => {
    if (!user) {
      setUnreadMessagesCount(0);
      return;
    }

    // Load recently read IDs from localStorage (with timestamps)
    const loadRecentlyRead = () => {
      try {
        const stored = localStorage.getItem(`sf_recently_read_${user.uid}`);
        if (stored) {
          const data = JSON.parse(stored) as { [id: string]: number }; // id -> timestamp
          const now = Date.now();
          // Remove entries older than 2 minutes
          Object.keys(data).forEach(id => {
            if (now - data[id] > 120000) { // 2 minutes
              delete data[id];
            } else {
              recentlyReadRef.current.add(id);
            }
          });
          // Save cleaned data back
          localStorage.setItem(`sf_recently_read_${user.uid}`, JSON.stringify(data));
        }
      } catch (e) {
        console.error('Failed to load recently read:', e);
      }
    };

    loadRecentlyRead();

    const fetchUnreadCount = async () => {
      try {
        const messages = await adminService.getSupportMessages(user.uid);

        // Count unread messages from admin (status !== 'read')
        // AND exclude locally read ones (optimistic update)
        const unread = messages.filter(m =>
          m.sender === 'admin' &&
          m.status !== 'read' &&
          !recentlyReadRef.current.has(m.id)
        ).length;

        // Cleanup ref: if server says it's read, remove from ref to keep it small
        messages.forEach(m => {
          if (m.status === 'read' && recentlyReadRef.current.has(m.id)) {
            recentlyReadRef.current.delete(m.id);
          }
        });

        setUnreadMessagesCount(unread);
      } catch (error) {
        console.error('Failed to fetch unread messages:', error);
      }
    };

    fetchUnreadCount();
    // Poll every 5 seconds
    const interval = setInterval(fetchUnreadCount, 5000);

    // Listen for optimistic updates from SettingsPage
    const handleMessagesRead = (e: Event) => {
      const customEvent = e as CustomEvent;
      const ids = customEvent.detail as string[];
      if (Array.isArray(ids)) {
        // Load current localStorage data
        const stored = localStorage.getItem(`sf_recently_read_${user.uid}`);
        const data = stored ? JSON.parse(stored) : {};
        const now = Date.now();

        ids.forEach(id => {
          recentlyReadRef.current.add(id);
          data[id] = now; // Store with timestamp
        });

        // Save to localStorage
        localStorage.setItem(`sf_recently_read_${user.uid}`, JSON.stringify(data));

        fetchUnreadCount(); // Re-calculate immediately
      }
    };
    window.addEventListener('sf_messages_read', handleMessagesRead);

    return () => {
      clearInterval(interval);
      window.removeEventListener('sf_messages_read', handleMessagesRead);
    };
  }, [user?.uid]); // Only re-run when user ID changes, not on every user property change

  // Poll for admin unread messages (from users)
  const [adminUnreadMessagesCount, setAdminUnreadMessagesCount] = useState(0);
  const recentlyReadAdminRef = useRef<Set<string>>(new Set());

  useEffect(() => {
    if (!user || user.role !== 'admin') {
      setAdminUnreadMessagesCount(0);
      return;
    }

    // Load recently read IDs from localStorage (with timestamps)
    const loadRecentlyRead = () => {
      try {
        const stored = localStorage.getItem(`sf_admin_recently_read_${user.uid}`);
        if (stored) {
          const data = JSON.parse(stored) as { [id: string]: number };
          const now = Date.now();
          // Remove entries older than 2 minutes
          Object.keys(data).forEach(id => {
            if (now - data[id] > 120000) { // 2 minutes
              delete data[id];
            } else {
              recentlyReadAdminRef.current.add(id);
            }
          });
          // Save cleaned data back
          localStorage.setItem(`sf_admin_recently_read_${user.uid}`, JSON.stringify(data));
        }
      } catch (e) {
        console.error('Failed to load admin recently read:', e);
      }
    };

    loadRecentlyRead();

    const fetchAdminUnreadCount = async () => {
      try {
        // Get ALL support messages
        const messages = await adminService.getSupportMessages();

        // Count unread messages from users (not from admin)
        // AND exclude locally read ones (optimistic update)
        const unread = messages.filter(m =>
          m.sender === 'user' &&
          m.status !== 'read' &&
          !recentlyReadAdminRef.current.has(m.id)
        ).length;

        // Cleanup ref
        messages.forEach(m => {
          if (m.status === 'read' && recentlyReadAdminRef.current.has(m.id)) {
            recentlyReadAdminRef.current.delete(m.id);
          }
        });

        setAdminUnreadMessagesCount(unread);
      } catch (error) {
        console.error('Failed to fetch admin unread messages:', error);
      }
    };

    fetchAdminUnreadCount();
    // Poll every 5 seconds
    const interval = setInterval(fetchAdminUnreadCount, 5000);

    // Listen for optimistic updates from AdminDashboard
    const handleAdminMessagesRead = (e: Event) => {
      const customEvent = e as CustomEvent;
      const ids = customEvent.detail as string[];
      if (Array.isArray(ids)) {
        // Load current localStorage data
        const stored = localStorage.getItem(`sf_admin_recently_read_${user.uid}`);
        const data = stored ? JSON.parse(stored) : {};
        const now = Date.now();

        ids.forEach(id => {
          recentlyReadAdminRef.current.add(id);
          data[id] = now; // Store with timestamp
        });

        // Save to localStorage
        localStorage.setItem(`sf_admin_recently_read_${user.uid}`, JSON.stringify(data));

        fetchAdminUnreadCount(); // Re-calculate immediately
      }
    };
    window.addEventListener('sf_admin_messages_read', handleAdminMessagesRead);

    return () => {
      clearInterval(interval);
      window.removeEventListener('sf_admin_messages_read', handleAdminMessagesRead);
    };
  }, [user]);

  // Handle Responsive sidebar behavior
  useEffect(() => {
    const handleResize = () => {
      if (window.innerWidth < 1024) {
        setIsSidebarCollapsed(false); // Reset collapse on mobile
      }
    };
    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);

  // Reset scroll on view change
  useEffect(() => {
    if (mainContentRef.current) {
      mainContentRef.current.scrollTo({ top: 0, behavior: 'instant' });
    }
  }, [currentView, appView, adminSection]);

  // Scroll detection for main content area
  useEffect(() => {
    const handleScroll = () => {
      if (mainContentRef.current) {
        setShowScrollTop(mainContentRef.current.scrollTop > 300);
      }
    };

    const mainEl = mainContentRef.current;
    if (mainEl) {
      mainEl.addEventListener('scroll', handleScroll);
    }

    return () => {
      if (mainEl) {
        mainEl.removeEventListener('scroll', handleScroll);
      }
    };
  }, [appView, currentView]); // Re-attach if view changes

  const scrollToTop = () => {
    if (mainContentRef.current) {
      mainContentRef.current.scrollTo({ top: 0, behavior: 'smooth' });
    }
  };

  const handleLogout = async () => {
    await authService.logout();
    setUser(null);
    setAppView(AppView.LANDING); // Redirect to Landing Page instead of Login
    setCurrentView(AppView.DASHBOARD);
  };

  const handleAuthSuccess = (loggedInUser: User) => {
    setUser(loggedInUser);
    setAppView(AppView.DASHBOARD);
    authService.checkDailyReset(loggedInUser.uid);
  };

  const handleOpenSettings = (tab: string = 'profile') => {
    setSettingsActiveTab(tab);
    setCurrentView(AppView.SETTINGS);
  };

  const getViewTitle = (view: AppView) => {
    switch (view) {
      case AppView.DASHBOARD: return 'Dashboard';
      case AppView.CHAT_INTELLIGENCE: return 'Market Intelligence Hub';
      case AppView.IMAGE_GENERATION: return 'Studio / Image Generator';
      case AppView.VIDEO_GENERATION: return 'Studio / Video Creator';
      case AppView.REVIEWER_AI: return 'Studio / Reviewer AI';
      case AppView.PROMPT_GENERATOR: return 'Studio / Prompt Generator';
      case AppView.MEDIA_ANALYSIS: return 'Metadata AI Analysis';
      case AppView.IMAGE_EDITOR: return 'Studio / Magic Editor';
      case AppView.LIBRARY: return 'Asset Library';
      case AppView.FAVORITES: return 'Favorites';
      case AppView.TRASH: return 'Trash';
      case AppView.SETTINGS: return 'Settings';
      case AppView.ADMIN: return 'Admin Console';
      default: return appConfig.appName || 'StockForge';
    }
  };

  const renderMainContent = () => {
    switch (currentView) {
      case AppView.DASHBOARD: return <Dashboard setView={setCurrentView} user={user} onOpenSettings={handleOpenSettings} />;
      case AppView.CHAT_INTELLIGENCE: return <MarketIntelligence />;
      case AppView.IMAGE_GENERATION: return <ImageGenerator />;
      case AppView.VIDEO_GENERATION: return <VideoGenerator />;
      case AppView.REVIEWER_AI: return <StockPhotoReviewer />;
      case AppView.PROMPT_GENERATOR: return <PromptGenerator />;
      case AppView.MEDIA_ANALYSIS: return <MetadataGenerator />;
      case AppView.IMAGE_EDITOR: return <ImageEditor />;
      case AppView.SETTINGS: return <SettingsPage user={user} activeTab={settingsActiveTab} onTabChange={setSettingsActiveTab} unreadMessagesCount={unreadMessagesCount} />;
      case AppView.ADMIN: return <AdminDashboard adminSection={adminSection} />;
      // Library Views
      case AppView.LIBRARY: return <Library view={AppView.LIBRARY} />;
      case AppView.FAVORITES: return <Library view={AppView.FAVORITES} />;
      case AppView.TRASH: return <Library view={AppView.TRASH} />;
      default: return <Dashboard setView={setCurrentView} user={user} onOpenSettings={handleOpenSettings} />;
    }
  };

  // Show loading screen while Firebase checks auth status
  if (authLoading) {
    return (
      <div className="min-h-screen bg-slate-50 dark:bg-slate-950 flex items-center justify-center">
        <div className="text-center">
          <div className="w-16 h-16 border-4 border-amber-500 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-slate-600 dark:text-slate-400 font-medium">Loading...</p>
        </div>
      </div>
    );
  }

  if (appView === AppView.LANDING) {
    return <LandingPage
      onStart={() => { setAuthView(AuthView.REGISTER); setAppView(AppView.AUTH); }}
      onSignIn={() => { setAuthView(AuthView.LOGIN); setAppView(AppView.AUTH); }}
    />;
  }

  if (appView === AppView.AUTH) {
    return <AuthPage
      initialView={authView}
      onAuthSuccess={handleAuthSuccess}
      onBackToLanding={() => setAppView(AppView.LANDING)}
    />;
  }

  return (
    <BatchProvider>
      <div className="flex h-screen bg-slate-50 dark:bg-[#020617] text-slate-900 dark:text-slate-200 overflow-hidden selection:bg-amber-500/30 font-sans transition-colors duration-300">
        <Sidebar
          currentView={currentView}
          setView={(view) => {
            // Reset settings tab to profile when navigating normally
            if (view === AppView.SETTINGS) setSettingsActiveTab('profile');
            setCurrentView(view);
          }}
          onLogout={handleLogout}
          user={user}
          isCollapsed={isSidebarCollapsed}
          toggleCollapse={() => setIsSidebarCollapsed(!isSidebarCollapsed)}
          isMobileOpen={isMobileSidebarOpen}
          closeMobile={() => setIsMobileSidebarOpen(false)}
          onOpenSettings={handleOpenSettings}
          onOpenDocumentation={() => handleOpenSettings('documentation')}
          // Admin Props
          adminSection={adminSection}
          setAdminSection={setAdminSection}
          // Config
          appName={appConfig.appName}
          appLogo={siteSettings?.appLogo || appConfig.appLogo}
          // Notification Badge
          unreadMessagesCount={user?.role === 'admin' ? adminUnreadMessagesCount : unreadMessagesCount}
          sidebarSettings={siteSettings?.sidebarSettings}
        />

        <div className="flex-1 flex flex-col min-w-0 bg-slate-50 dark:bg-slate-950 relative transition-colors duration-300">
          {/* Ambient Background */}
          <div className="absolute inset-0 pointer-events-none bg-[radial-gradient(ellipse_at_top,_var(--tw-gradient-stops))] from-slate-200/40 dark:from-slate-800/10 via-transparent to-transparent z-0"></div>

          {/* Admin View As Banner */}
          {isViewingAsUser && user && (
            <div className="bg-amber-500 text-black px-4 py-2 font-bold flex items-center justify-between shadow-md relative z-50">
              <div className="flex items-center gap-2">
                <span className="bg-black/20 p-1 rounded"><ScanEye size={16} /></span>
                <span>Viewing Dashboard as: {user.displayName || user.email} ({user.uid})</span>
              </div>
              <button
                onClick={() => {
                  // Remove param and reload to return to admin
                  const url = new URL(window.location.href);
                  url.searchParams.delete('admin_view_as');
                  window.location.href = url.toString();
                }}
                className="bg-black text-white px-3 py-1 rounded text-sm hover:bg-slate-800 transition-colors"
              >
                Exit View
              </button>
            </div>
          )}

          <TopBar
            title={getViewTitle(currentView)}
            toggleMobileSidebar={() => setIsMobileSidebarOpen(!isMobileSidebarOpen)}
            user={user}
            onLogout={handleLogout}
            setView={setCurrentView}
            onOpenDocumentation={() => handleOpenSettings('documentation')}
            onOpenSettings={handleOpenSettings}
            appName={appConfig.appName}
            unreadMessagesCount={user?.role === 'admin' ? adminUnreadMessagesCount : unreadMessagesCount}
          />

          <main className="flex-1 overflow-y-auto relative z-10" ref={mainContentRef}>
            {renderMainContent()}
          </main>

          {/* Global Scroll To Top Button for App Views */}
          <button
            onClick={scrollToTop}
            className={`fixed bottom-8 right-8 p-2 bg-gradient-to-r from-amber-400 to-orange-500 hover:from-amber-300 hover:to-orange-400 text-slate-900 rounded-full shadow-lg shadow-orange-500/20 z-50 transition-all duration-300 transform hover:scale-110 ${showScrollTop ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-10 pointer-events-none'
              }`}
          >
            <ArrowUp size={20} strokeWidth={3} />
          </button>
        </div>

      </div>
    </BatchProvider>
  );
}

export default App;
